//=============================================================================
// File: Pneumatics.cpp
// Desc: Declares some simple classes to aid with pressure calculations for
//       pneumatic brake pipes, reservoirs, etc.
//=============================================================================
#include "Pneumatics.h"
#include "TNIPhysicsGlobals.h"
#include "UnitConversion.h"



//=============================================================================
// Name: SanityCheckMass
// Desc: Called regularly to sanity check that a pipe mass value hasn't been
//       corrupted, either by save/load, or invalid internal maths.
// Parm: mass - A pipe mass value to sanity check.
//=============================================================================
static void SanityCheckMass(double mass)
{
  NANCheck(mass);
  ASSERT_ONCE(mass >= 0, s_bSanityCheckMassk_Negative);
}


//=============================================================================
static const double kExchangeRate = 100;
static const double kSmallFloat = 0.000001;


//=============================================================================
// Name: Pneumatic
// Desc: Constructor for a new pneumatic pipe/reservoir. Takes the volume as
//       a constructing param, as changes to volume are not support.
// Parm: volume - The volume of this pipe/reservoir, in cubic metres (m^3). If
//       a programmer wishes to disable/nullify a particular pipe/system, then
//       this may be passed as zero (see also: Pneumatic::Exists()).
//=============================================================================
Pneumatic::Pneumatic(double volume)
  : m_volume(volume)
  , m_mass(volume * kAtmosphericPressureGM3)
  , m_ignoreMass(false)
{
  SanityCheckMass(m_mass);
}


//=============================================================================
// Name: GetPressure
// Desc: Returns the pressure of this volume.
// Retn: double - The pressure of this volume, in g/m^3.
// Note: See UnitConversion.h for conversion functions to PSI/kpa if required.
//=============================================================================
double Pneumatic::GetPressure(void) const
{
  SanityCheckMass(m_mass);

  if (m_volume <= 0.0f)
    return kAtmosphericPressureGM3;

  return m_mass / m_volume;
}


//=============================================================================
// Name: SetPressure
// Desc: Sets the pressure of this volume.
// Parm: pressure - the pressure of this volume, in g/m^3.
// Note: See UnitConversion.h for functions to aquire this value from PSI/kpa.
//=============================================================================
void Pneumatic::SetPressure(double pressure)
{
  if (m_volume >= 0.0f)
    m_mass = m_volume * pressure;

  SanityCheckMass(m_mass);
}


//=============================================================================
// Name: Exists
// Desc: Returns whether this object actually has any volume. This is provided
//       as a way to disable/nullify a particular system on a train vehicle.
// Retn: bool - True if this object has a valid volume.
//=============================================================================
bool Pneumatic::Exists(void) const
{
  SanityCheckMass(m_mass);
  return m_volume > 0;
}


//=============================================================================
// Name: Exchange
// Desc: Exchanges gas between two volumes, based on the pressure of each, over
//       the time period passed.
// Parm: target - (IN, OUT) - The volume to exchange with, if possible.
// Parm: flowSize - The rate of flow between these two volumes (from the train
//       vehicle asset spec).
// Parm: dt - The time period to exchange for (in seconds).
// Parm: pressureRatio - Pressure conversion rate for coupling cars with
//       different braking pressures. This is set to the asset spec pressure of
//       this pipe divided by the spec pressure of the other.
//=============================================================================
void Pneumatic::Exchange(Pneumatic& target, double flowSize, float dt, double pressureRatio)
{
  if (!Exists() || !target.Exists())
    return;

  const double myPressure = GetPressure();
  const double tgPressure = target.GetPressure() * pressureRatio;
  const double diff = tgPressure - myPressure;

  double exchange = (diff * fabs(diff)) * flowSize * kExchangeRate * dt;
  NANCheck(exchange);

  // Don't overshoot into negative mass.
  if (!m_ignoreMass)
  {
    if (-exchange > m_mass)
      exchange = -m_mass;
  }
  if (!target.m_ignoreMass)
  {
    if (exchange > target.m_mass * pressureRatio)
      exchange = target.m_mass * pressureRatio;
  }

  if (!m_ignoreMass)
    m_mass += exchange;
  if (!target.m_ignoreMass)
    target.m_mass -= exchange / pressureRatio;

  // Don't allow us to overshoot the target pressure in this time interval.
  if (fabs(GetPressure() - myPressure) > fabs(diff * 0.5f))
  {
    SetPressure(myPressure + diff * 0.5f);
    target.SetPressure((myPressure + diff * 0.5f) / pressureRatio);
  }

  // If the mass value has gone out of bounds due to numerical inaccuracy,
  // just round it to zero.
  if (m_mass < 0 && m_mass > -kSmallFloat)
    m_mass = 0;
  if (target.m_mass < 0 && target.m_mass > -kSmallFloat)
    target.m_mass = 0;


  //
  SanityCheckMass(m_mass);
  SanityCheckMass(target.m_mass);
}


//=============================================================================
// Name: EqualisedExchange
// Desc: Exchanges gas between two volumes, based on the pressure of each, over
//       the time period passed.
// Parm: target - (IN, OUT) - The volume to exchange with, if possible.
// Parm: flowSize - The rate of flow between these two volumes (from the train
//       vehicle asset spec).
// Parm: dt - The time period to exchange for (in seconds).
// Parm: cutOffPressure - A cutoff value which specifies a maximum pressure to
//       exchange to. Prevents this volume becoming over-pressurised.
//=============================================================================
void Pneumatic::EqualisedExchange(Pneumatic& target, double flowSize, float dt, double cutOffPressure)
{
  if (!Exists() || !target.Exists())
    return;

  const double myPressure = GetPressure();
  const double tgPressure = target.GetPressure();
  const double diff = tgPressure - myPressure;

  double exchange = (diff * fabs(diff)) * flowSize * kExchangeRate * dt;
  NANCheck(exchange);

  // Check if this exchange would overshoot the target pressure, and clamp it if so.
  if (diff > 0)
  {
    if ((m_mass + exchange) / m_volume > cutOffPressure)
      exchange = (m_volume * cutOffPressure) - m_mass;
  }
  else
  {
    if ((m_mass + exchange) / m_volume < cutOffPressure)
      exchange = (m_volume * cutOffPressure) - m_mass;
  }

  // Don't overshoot into negative mass.
  if (!m_ignoreMass)
  {
    if (-exchange > m_mass)
      exchange = -m_mass;
  }
  if (!target.m_ignoreMass)
  {
    if (exchange > target.m_mass)
      exchange = target.m_mass;
  }

  if (!m_ignoreMass)
    m_mass += exchange;
  if (!target.m_ignoreMass)
    target.m_mass -= exchange;

  // Don't allow us to overshoot the target pressure in this time interval.
  if (fabs(GetPressure() - myPressure) > fabs(diff * 0.5f))
  {
    SetPressure(myPressure + diff * 0.5f);
    target.SetPressure(myPressure + diff * 0.5f);
  }

  // If the mass value has gone out of bounds due to numerical inaccuracy,
  // just round it to zero.
  if (m_mass < 0 && m_mass > -kSmallFloat)
    m_mass = 0;
  if (target.m_mass < 0 && target.m_mass > -kSmallFloat)
    target.m_mass = 0;


  //
  SanityCheckMass(m_mass);
  SanityCheckMass(target.m_mass);
}


//=============================================================================
// Name: FixedPneumatic
// Desc: Constructor for a new fixed pneumatic source. Takes the pressure as
//       a constructing param, as changes to this are not support.
// Parm: pressure - The pressure of this pneumatic source, in grams per cubic
//       metre (g/m^3). If a programmer wishes to disable/nullify a particular
//       system, then this may be zero (see also: Pneumatic::Exists()).
//=============================================================================
FixedPneumatic::FixedPneumatic(double pressure)
  : Pneumatic(0)
  , m_pressure(pressure)
{
  m_ignoreMass = true;
  SanityCheckMass(m_pressure);
}


