//=============================================================================
// File: Pneumatics.h
// Desc: Declares some simple classes to aid with pressure calculations for
//       pneumatic brake pipes, reservoirs, etc.
//=============================================================================
#ifndef _Pneumatics_H_
#define _Pneumatics_H_
#pragma once



//=============================================================================
// Name: Pneumatic
// Desc: Basically a code representation of an air tank/pipe. This structure
//       has a volume, and a mass, from which we can derive pressure. It also
//       has accessors to aid with exchanging air between two structures.
// Note: This class makes various assumptions based on ideal gas law (including
//       constant temperature), and returns/takes pressure values as g/m^3
//       (technically a unit of density, not pressure). See UnitConversion.h
//       for appropriate conversion functions.
//=============================================================================
class Pneumatic
{
public:

  Pneumatic(double volume);

  virtual void SetPressure(double pressure);
  virtual double GetPressure(void) const;
  virtual bool Exists(void) const;

  void Exchange(Pneumatic& target, double flowSize, float dt, double pressureRatio = 1.0);
  void EqualisedExchange(Pneumatic& target, double flowSize, float dt, double cutOffPressure);


protected:

  double  m_volume;         // Cubic metres (m^3)
  double  m_mass;           // Grams (g)
  bool    m_ignoreMass;     // Set by FixedPneumatic, used by the exchange functions.

};


//=============================================================================
// Name: FixedPneumatic
// Desc: Similar to Pneumatic, but for a fixed pressure source without a
//       measurable/quantifiable mass. For example, an air compressor, an
//       exaust, or the atmosphere itself.
//=============================================================================
class FixedPneumatic : public Pneumatic
{
public:

  FixedPneumatic(double pressure);

  virtual double GetPressure(void) const override { return m_pressure; }
  virtual bool Exists(void) const override { return m_pressure > 0; }


protected:

  double  m_pressure;       // Grams per cubic metre (g/m^3)

};



#endif // _Pneumatics_H_

