//=============================================================================
// File: SteamPhysics_Boiler.cpp
// Desc: Defines SteamPhysics::Boiler, an object representing the boiler
//       within a steam engine.
//=============================================================================
#include "SteamPhysicsComponents.h"
#include "TNIFunctions.hpp"
#include "UnitConversion.h"


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{

//=============================================================================
#define GetBoilingTemperature(bar)  (29.719 * log(bar) + 99.63 + bar + 273) // Kelvin
#define GetKilojoulestoBoil(bar)    (125.8 * log(bar) + 417.51 + 5.7 * (bar >= 2 ? bar - 2 : 0))
#define GetJoulestoBoil(bar)        (GetKilojoulestoBoil(bar) * 1000)
#define GetConversionEnergy(bar)    (-81.267 * log(bar) + 2257.92 - 5.7 * (bar >= 4 ? bar : 0)) // KiloJoules



//=============================================================================
// Name: Boiler
// Desc: Steam engine boiler constructor. Takes a number of required values
//       which specify how the boiler behaves (all pulled from the engine
//       asset spec), and sets everything else to the defaults.
// Parm: boilerVol - The volume of the boiler, in cubic metres (m^3).
// Parm: initialWaterMass - The starting mass of the water in the boiler, in
//       Kilograms. Water is under pressure within the boiler, so mass is used
//       over volume as it remains constant.
// Parm: initialWaterTempInK - The starting temperature of the water in the
//       boiler, in Kelvin.
// Parm: initialSteamMass - The mass of the steam in the boiler, in Kilograms.
// Parm: boilerEfficiencyIdle - The idle efficiency of the boiler, from 0 to 1.
// Parm: boilerEfficiencyTest - The 'normal'/tested efficiency of the boiler
//       when actually in use, from 0 to 1.
// Parm: boilerEfficiencyMin - The absolute minimum efficiency of the boiler,
//       from 0 to 1.
// Parm: boilerHeatLoss - Rate at which heat is radiated away from the boiler
//       and into the environment.
//=============================================================================
Boiler::Boiler(double boilerVol, double initialWaterMass,
               double initialWaterTempInK, double initialSteamMass,
               double boilerEfficiencyIdle, double boilerEfficiencyTest,
               double boilerEfficiencyMin, double boilerHeatLoss)
{
  m_waterMass = initialWaterMass;
  m_waterTemp = initialWaterTempInK;
  m_steamMass = initialSteamMass;
  m_steamTemp = m_waterTemp;
  m_storedEnergy = 0;
  m_previousPressure = kAtmosphericPressurePascals;

  m_specBoilerVolume = boilerVol;
  m_specBoilerHeatLoss = boilerHeatLoss;

  // These efficiency values specify the percentage at which the input energy
  // is used to convert water into steam (i.e, 1.0-boilerEfficiencyIdle is how
  // much energy is lost).
  m_specEfficiencyIdle = boilerEfficiencyIdle;
  m_specEfficiencyTest = boilerEfficiencyTest;
  m_specEfficiencyMin = boilerEfficiencyMin;
}


//=============================================================================
// Name: AddWater
// Desc: Adds water to the boiler. This would typically be done based on the
//       players injector settings (or similarly fudged values for DCC mode).
// Parm: amount - The mass of water to add, in Kilograms.
// Parm: temperature - The temperature of the water being added, in Kelvin.
// Retn: double - The mass of water (kg) actually added. This will differ from
//       'addAmount' if the caller tried to fill the boiler beyond 99% volume
//       (for current density, as per temperature and pressure).
//=============================================================================
double Boiler::AddWater(double addAmount, float addTemp)
{
  // Calculate current water density in kg/m^3.
  double density = CalculateWaterDensity();
  ASSERT(density > 0);

  // Calculate the space available, capping at 99% to avoid a hydraulic lock,
  // but otherwise ignoring the steam (we'll force it into the pistons or out
  // a safety valve as necessary).
  double availableVolume = (m_specBoilerVolume * 0.99) - (m_waterMass / density);

  if (addAmount / density > availableVolume)
  {
    // The requested water mass not fit at current density. Calculate how much
    // we *can* fit.
    addAmount = availableVolume * density;
    ASSERT(addAmount >= 0);
  }

  // Adjust the temperature for the incoming water, then add in the water amount.
  m_waterTemp = (m_waterMass * m_waterTemp + addAmount * addTemp) / (m_waterMass + addAmount);
  m_waterMass += addAmount;

  // Return the amount of water actually added, which the calling code will use
  // to update the loco/tender product queues.
  return addAmount;
}


//=============================================================================
// Name: RemoveSteam
// Desc: Attempts to remove the specified amount of steam from the boiler,
//       returning the actual amount removed.
// Parm: amount - The mass of steam (kg) to attempt to remove from the boiler.
// Retn: double - The mass of steam (kg) actually removed. This will differ
//       from 'amount' if the caller requested more steam than was available.
//=============================================================================
double Boiler::RemoveSteam(double amount)
{
  if (amount < 0)
    amount = 0;
  else if (amount > m_steamMass)
    amount = m_steamMass;

  m_steamMass -= amount;
  return amount;
}


//=============================================================================
// Name: Update
// Desc: Updates the boiler for a given time interval. This uses the stored
//       energy to boil stored water into steam, as appropriate for the current
//       internal pressure.
// Parm: dt - The time interval over which to update, in seconds.
// Parm: blowerSetting - The current firebox blower setting.
// Parm: blowerSteamDemand - The rate at which we lose steam mass via the
//       blower, as determined from the engine asset spec.
// Parm: efficiencyFactor - The currently determined efficiency of the boiler.
//       This may be calculated based on other engine settings/state, engine
//       wear, environmental conditions, etc.
//=============================================================================
void Boiler::Update(float dt, double blowerSetting, double blowerSteamDemand, double efficiencyFactor)
{
  m_steamTemp = m_waterTemp;
  m_previousPressure = GetBoilerPressure();

  // Radiate heat away from the boiler.
  if (m_waterMass > 0)
  {
    // External temperature of 27C assumed.
    m_waterTemp -= (m_waterTemp - 290) * dt * m_specBoilerHeatLoss / m_waterMass;
    NANCheck(m_waterTemp);
  }
  else
  {
    m_waterTemp = 290;
  }

  // Calculate the boiling temperature for current pressure
  double curBoilingTemp = GetBoilingTemperature(UnitConversion::pa_bar(GetBoilerPressure()));

  double efficiency = m_specEfficiencyIdle + (m_specEfficiencyTest - m_specEfficiencyIdle) * efficiencyFactor;
  if (efficiency < m_specEfficiencyMin)
    efficiency = m_specEfficiencyMin;

  m_storedEnergy *= efficiency;

  double energyNeededToBringWaterToBoil = (curBoilingTemp - m_waterTemp) / (curBoilingTemp - 273) *
                                          GetJoulestoBoil(UnitConversion::pa_bar(GetBoilerPressure())) * m_waterMass;
  NANCheck(energyNeededToBringWaterToBoil);


  if (energyNeededToBringWaterToBoil < 0)
  {
    // The water is above boiling temp, cool it down a bit, and store the
    // energy released (which we'll use below to create steam). Rate is very
    // low to stop spikey behaviour of pressure.
    m_waterTemp += (curBoilingTemp - m_waterTemp) * dt * 0.05;
    m_storedEnergy -= energyNeededToBringWaterToBoil * dt * 0.05;
  }
  else if (m_storedEnergy > 0 && energyNeededToBringWaterToBoil > m_storedEnergy)
  {
    // We have energy, but not enough to boil the water. Use it all to raise the
    // water temperature as much as we can.
    m_waterTemp += m_storedEnergy / energyNeededToBringWaterToBoil * (curBoilingTemp - m_waterTemp);
    m_storedEnergy = 0;
  }
  else if (m_storedEnergy > 0)
  {
    // We have enough energy to boil the water, do it.
    m_waterTemp = curBoilingTemp;
    m_storedEnergy -= energyNeededToBringWaterToBoil;
  }
  else
  {
    // Unhandled; what to do if the fire is cooler than the boiler.
  }


  // If there's any energy left after heating the water then we'll be at boiling
  // temperature, and can convert some water to steam.
  if (m_storedEnergy > 0)
  {
    // Calculate the conversion rate for transitioning the water to steam.
    double conversionEnergy = GetConversionEnergy(UnitConversion::pa_bar(GetBoilerPressure())) * 1000;

    // 2008-04-08: Added rate limit to squash odd effects from negative
    // conversion energy at very high pressures.
    if (conversionEnergy < 1)
      conversionEnergy = 1;

    double massToConvert = m_storedEnergy / conversionEnergy;
    NANCheck(massToConvert);
    if (m_waterMass < massToConvert)
      massToConvert = m_waterMass;

    m_waterMass -= massToConvert;
    m_steamMass += massToConvert;
    m_storedEnergy = 0;
  }

  // Normalise the boiler temperature
  m_steamTemp = m_waterTemp;

  // Lose some steam via the blower
  m_steamMass -= (blowerSteamDemand * blowerSetting * dt); // proportion of "full" draft per second.

  // TODO: Should remove some steam from the boiler while injectors are on too
}


//=============================================================================
// Name: GetBoilerPressure
// Desc: Returns the current steam pressure within the boiler, in Pascals.
// Retn: double - The pressure of the steam within the boiler, in Pascals.
//=============================================================================
double Boiler::GetBoilerPressure(void)
{
  // 55.6 is mol/kg water (normally seen as 18.02 g/mol)
  double pressure = m_steamMass * 55.6 * RCONST * m_steamTemp / GetBoilerSteamVolume();
  NANCheck(pressure);

  // Prevent the boiler dipping below atmospheric pressure somehow
  if (pressure < kAtmosphericPressurePascals)
  {
    m_steamMass = (kAtmosphericPressurePascals * GetBoilerSteamVolume()) / (55.6 * RCONST * m_steamTemp);
    NANCheck(m_steamMass);
    pressure = kAtmosphericPressurePascals;
  }

  return pressure;
}


//=============================================================================
// Name: GetBoilerPressurePSI
// Desc: Returns the current pressure within the boiler, in PSI.
// Retn: double - The pressure within the boiler, in PSI.
//=============================================================================
double Boiler::GetBoilerPressurePSI(void)
{
  return UnitConversion::pa_psi(GetBoilerPressure());
}


//=============================================================================
// Name: GetBoilerWaterVolume
// Desc: Returns the volume of water in the boiler, based on current mass and
//       and density (as derived from temperature and pressure).
// Parm: double - The volume of water in the boiler, in cubic metres (m^3).
//=============================================================================
double Boiler::GetBoilerWaterVolume(void) const
{
  // Calculate current water density in kg/m^3.
  double density = CalculateWaterDensity();
  ASSERT(density > 0);
  return m_waterMass / density;
}


//=============================================================================
// Name: GetBoilerSteamVolume
// Desc: Returns the volume of steam in the boiler.
// Parm: double - The volume of steam in the boiler, in cubic metres (m^3).
//=============================================================================
double Boiler::GetBoilerSteamVolume(void) const
{
  // For simplicity, we assume the boiler always contains only water and steam.
  double volume = m_specBoilerVolume - GetBoilerWaterVolume();

  // Sanitise this greater than zero, as we divide by it to get pressure.
  if (volume <= 0)
  {
    ASSERT_ONCE(volume >= 0, s_bBoiler_GetBoilerSteamVolume_Negative);
    return 0.1;
  }

  return volume;
}


//=============================================================================
// Name: CalculateWaterDensity
// Desc: Calculates the density of the water in the boiler, based on internal
//       pressure and temperature.
// Retn: double - The density of the water in the boiler, in Kilograms per
//       cubic metre (kg/m^3).
//=============================================================================
double Boiler::CalculateWaterDensity(void) const
{
  // The density of a fluid when changing both temperature and pressure can be
  // expressed as:
  //    ?1 = [ ?0 / (1 +  (t1 - t0)) ] / [1 - (p1 - p0) * E]
  //
  // where:
  //    ?1 = final density (kg/m3)
  //    ?0 = initial density (kg/m3)
  //     = volumetric temperature expansion coefficient (m3/m3 oC)
  //    t1 = final temperature (oC)
  //    t0 = initial temperature (oC)
  //    E = bulk modulus fluid elasticity (N/m2)
  //    p1 = final pressure (N/m2)
  //    p0 = initial pressure (N/m2)
  //
  // Volumetric Temperature Coefficients -  
  //    * water : 0.88 10-4 (m3/m3 oC)
  //    * ethyl alcohol : 110 10-5 (m3/m3 oC)
  //    * oil : 63.3 10-5 (m3/m3 oC)
  //
  // Bulk Modulus Fluid Elasticity some common Fluids - E
  //    * water : 2.15 109 (N/m2)
  //    * ethyl alcohol : 1.06 109 (N/m2)
  //    * oil : 1.5 109 (N/m2)

  // Calculate density for water, given known density of 1000 kg/m^3 at
  // atmospheric pressure and around 4 degrees.
  double density = (1000.0 / (1 + (0.88E-4) * (m_waterTemp - 277))) / (1 - (m_previousPressure - kAtmosphericPressurePascals) * (2.15109E-9));
  NANCheck(density);
  return density;
}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the boiler to the soup passed. Save will
//       always use the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the boiler state into. A sub-soup will be
//       created for this purpose, to limit the chance of name overlap.
//=============================================================================
void Boiler::Save(TNIRef<TNISoup>& io_data) const
{
  TNIRef<TNISoup> boilerData = TNIAllocSoup();
  SoupSetFloat(boilerData, g_strings->lblTagWaterMass, m_waterMass);
  SoupSetFloat(boilerData, g_strings->lblTagWaterTemp, m_waterTemp);
  SoupSetFloat(boilerData, g_strings->lblTagSteamMass, m_steamMass);
  SoupSetFloat(boilerData, g_strings->lblTagSteamTemp, m_steamTemp);
  SoupSetFloat(boilerData, g_strings->lblTagEnergy, m_storedEnergy);
  SoupSetFloat(boilerData, g_strings->lblTagPrevPressure, m_previousPressure);

  TNISetSoupKeyValue(io_data, g_strings->lblTagBoiler, boilerData);

  // Everything else is from the spec, and doesn't need to be saved.
}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the boiler from the soup passed.
// Parm: data - The (read-only) soup to restore the boiler state from. Note
//       that this soup may not have been created by this plugin.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void Boiler::Load(const TNIRef<const TNISoup>& data, int dataVersion)
{
  const TNIRef<const TNISoup> boilerData = TNICastSoup(TNIGetSoupValueByKey(data, g_strings->lblTagBoiler));
  if (!boilerData)
    return;

  m_waterMass = SoupGetFloat(boilerData, g_strings->lblTagWaterMass, m_waterMass);
  m_waterTemp = SoupGetFloat(boilerData, g_strings->lblTagWaterTemp, m_waterTemp);
  m_steamMass = SoupGetFloat(boilerData, g_strings->lblTagSteamMass, m_steamMass);
  m_steamTemp = SoupGetFloat(boilerData, g_strings->lblTagSteamTemp, m_steamTemp);
  m_storedEnergy = SoupGetFloat(boilerData, g_strings->lblTagEnergy, m_storedEnergy);
  m_previousPressure = SoupGetFloat(boilerData, g_strings->lblTagPrevPressure, m_previousPressure);

  // Sanity check the ranges on these variables.
  m_waterMass = SanityCheckClamp(0, m_waterMass, m_waterMass);
  m_waterTemp = SanityCheckClamp(273, m_waterTemp, m_waterTemp);
  m_steamMass = SanityCheckClamp(0, m_steamMass, m_steamMass);
  m_steamTemp = SanityCheckClamp(273, m_steamTemp, m_steamTemp);
  m_storedEnergy = SanityCheckClamp(0, m_storedEnergy, m_storedEnergy);
  m_previousPressure = SanityCheckClamp(kAtmosphericPressurePascals, m_previousPressure, m_previousPressure);
}


}; // namespace SteamPhysics

