//=============================================================================
// File: SteamPhysics_Boiler.h
// Desc: Declares SteamPhysics::Boiler, an object representing the boiler
//       within a steam engine.
//=============================================================================
#ifndef _SteamPhysics_Boiler_H_
#define _SteamPhysics_Boiler_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{


//=============================================================================
// Name: Boiler
// Desc: The steam engine boiler. This takes energy from the firebox and uses
//       it to convert water into steam at high pressure.
//=============================================================================
class Boiler
{
public:

  Boiler( double boilerVol, double initialWaterMass, double initialWaterTempinK,
          double initialSteamMass, double boilerEfficiencyIdle, double boilerEfficiencyTest,
          double boilerEfficiencyMin, double boilerHeatLoss);

  double AddWater(double amount, float temp = 350);
  void AddEnergy(double amount) { m_storedEnergy += amount; }
  double RemoveSteam(double amount);

  void Update(float dt, double blowerSetting, double blowerSteamDemand, double efficiencyFactor);

  double GetBoilerVolume(void) const { return m_specBoilerVolume; }
  double GetBoilerPressure(void);
  double GetBoilerPressurePSI(void);

  double GetBoilerWaterVolume(void) const;
  double GetBoilerWaterMass(void) const { return m_waterMass; }
  double GetBoilerWaterTemp(void) const { return m_waterTemp; }

  double GetBoilerSteamVolume(void) const;
  double GetBoilerSteamMass(void) const { return m_steamMass; }
  double GetBoilerSteamTemp(void) const { return m_steamTemp; }


  void Save(TNIRef<TNISoup>& io_data) const;
  void Load(const TNIRef<const TNISoup>& data, int dataVersion);


private:

  double CalculateWaterDensity(void) const;


  // These values are tracked and updated between frames, and need to be saved/loaded.
  double m_waterMass;           // Kilograms
  double m_waterTemp;           // Degrees Kelvin
  double m_steamMass;           // Kilograms
  double m_steamTemp;           // Degrees Kelvin
  double m_storedEnergy;        // Joules
  double m_previousPressure;    // Pascals


  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.

  double m_specBoilerVolume;    // Cubic metres (m^3)
  double m_specEfficiencyIdle;  // Idle boiler efficiency 0 to 1
  double m_specEfficiencyTest;  // Test/'normal' efficiency 0 to 1
  double m_specEfficiencyMin;   // Absolute minimum efficiency, 0 to 1
  double m_specBoilerHeatLoss;  // Rate at which heat is radiated away from
                                // the boiler into the environment

};


}; // namespace SteamPhysics


#endif // _SteamPhysics_Boiler_H_

