//=============================================================================
// File: SteamPhysics_DrivingWheel.cpp
// Desc: Defines SteamPhysics::DrivingWheel, an object representing the
//       driving wheels within a steam engine.
//=============================================================================
#include "SteamPhysicsComponents.h"
#include "TNIFunctions.hpp"
#include "UnitConversion.h"
#include <algorithm>


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{


//=============================================================================
// Name: DrivingWheel
// Desc: Steam driving wheel constructor. This takes a number of required
//       values which specify how the wheel behaves (all pulled from the engine
//       asset spec), and sets everything else to the defaults. Accessors are
//       provided to alter other spec values, see the class declaration.
// Parm: vehicleMass - The initial mass of the train vehicle, in kilograms.
// Parm: wheelDiameter - The diameter of the driving wheel, in metres.
// Parm: drivingWheelWeightRatio - Ratio of the weight of this vehicle on the
//       driven wheels, from 0.0 (none) to 1.0 (full vehicle mass).
// Parm: brakeFriction - The friction caused by the main brakes when active.
// Parm: brakeRatio - Coefficient between the pressure in the brake cylinder
//       and the final brake force.
// Parm: handbrakeMaxForce - Handbrake force when fully active, in Newtons.
//=============================================================================
DrivingWheel::DrivingWheel(double vehicleMass, double wheelDiameter, double drivingWheelWeightRatio,
                           double brakeRatio, double handbrakeMaxForce)
  : m_pistonCount(0)
  , m_position(0)
  , m_angVel(0)
  , m_forwardMomentum(0)
  , m_bWheelSlipping(false)
  , m_statEngineForce(0)
  , m_statAppliedForce(0)
  , m_statWheelslipForce(0)
  , m_statBrakeForce(0)
  , m_statResistanceForce(0)
  , m_sandingFactor(0)
  , m_handBrakeSetting(0)
  , m_brakePressurePSI(0)
  , m_vehicleMass(std::max(1.0, vehicleMass))
  , m_currentGrade(0)
  , m_trackCurvature(0)
  , m_weightTE(0)
  , m_wheelSlipTractionMultiplier(0.25)
  , m_wheelSlipMomentumMultiplier(0.13)
  , m_specWheelDiameter(wheelDiameter)
  , m_specDrivingWheelWeightRatio(drivingWheelWeightRatio)
  , m_specBrakeFriction(0.3 * brakeRatio / 50)
  , m_specHandBrakeMaxForce(handbrakeMaxForce)
  , m_specMaximumSpeedHack(0)
  , m_specAxlecount(4)
  , m_specSurfaceArea(120.0)
  , m_specMovingFrictionCoefficient(0.03)
  , m_specAirDragCoefficient(0.0017)
{
  ASSERT(vehicleMass > 0 && wheelDiameter > 0);
}


//=============================================================================
// Name: AddPiston
// Desc: Adds the piston passed to the internal array. The wheel references
//       the pistons only, and does not own/delete them.
// Parm: piston - The piston to add to the internal array. The caller retains
//       ownership of this object, and must delete it themselves (ideally after
//       the wheel, or at least after the wheel is last used.
// Parm: wheelOffset - The angular offset of the pistons position on the wheel,
//       in radians, 0 to 2PI.
// Note: See STEAMPHYSICS_MAX_PISTONS for the maximum number of pistons.
//=============================================================================
void DrivingWheel::AddPiston(Piston* piston, double wheelOffset)
{
  if (m_pistonCount >= STEAMPHYSICS_MAX_PISTONS)
  {
    ASSERT(false);
    return;
  }

  m_pistons[m_pistonCount] = PistonInfo(piston, wheelOffset);
  ++m_pistonCount;
}


//=============================================================================
// Name: InitDavisFormulaValues
// Desc: Initialises the values required for davis formula calculations (for
//       moving track resistance). Called during wheel construction.
// NOTE: The Davis formula was empirically developed in America, and uses US
//       imperial units. Take care to perform appropriate conversions.
// Parm: axleCount - The total number of axles on the steam loco (for moving
//       friction calculations).
// Parm: surfaceArea - The forward facing surface area of the locomotive, in
//       square feet.
// Parm: movingFrictionCoefficient - Friction coefficient used to scale train
//       movement resistance based on speed.
// Parm: airDragCoefficient - Friction coefficient used to scale train movement
//       resistance based on speed squared, as appropriate for the aerodynamics
//       component of the resistance calculation.
//=============================================================================
void DrivingWheel::InitDavisFormulaValues(int axleCount, double surfaceArea, double movingFrictionCoefficient, double airDragCoefficient)
{
  m_specAxlecount = axleCount;
  m_specSurfaceArea = surfaceArea;
  m_specMovingFrictionCoefficient = movingFrictionCoefficient;
  m_specAirDragCoefficient = airDragCoefficient;
}


//=============================================================================
// Name: Update
// Desc: Calculates the current force from the pistons and applies it to the
//       wheel as velocity change. That force will also undergo wheelslip
//       checks, and some or all the wheel movement will be applied as traction
//       and adjust the forward momentum.
//       Brake and track resistance are also applied here, either directly to
//       the wheel movement or to the train movement (during wheel slip).
// Parm: dt - Time period to update the wheel for, in seconds.
//=============================================================================
void DrivingWheel::Update(float dt)
{
  const double wasVel = GetWheelVelocity();
  const double wheelRadius = m_specWheelDiameter / 2.0;

  m_statEngineForce = 0;
  m_statAppliedForce = 0;
  m_statWheelslipForce = 0;
  m_statBrakeForce = 0;
  m_statResistanceForce = 0;

  // Calculate the cumulative force from the pistons.
  double pushTorque = 0;
  for (int i = 0; i < m_pistonCount; ++i)
  {
    double force = m_pistons[i].m_pist->CalculateForce() * cos(m_position + m_pistons[i].offsetOnWheel);
    pushTorque += force * m_pistons[i].m_pist->GetPistonStrokeLength() / 2.0;

    m_statEngineForce += force;
  }

  m_statEngineForce = fabs(m_statEngineForce);

  // Check if that force exceeds the maximum tractive effort.
  double maxTE = m_weightTE * m_sandingFactor * m_specDrivingWheelWeightRatio;
  if (fabs(pushTorque / wheelRadius) > maxTE)
  {
    m_statWheelslipForce = fabs(pushTorque / wheelRadius) - maxTE;
    m_bWheelSlipping = true;
  }
  else if (m_bWheelSlipping)
  {
    if (fabs(pushTorque / wheelRadius) > maxTE * m_wheelSlipTractionMultiplier)
      m_statWheelslipForce = fabs(pushTorque / wheelRadius) - maxTE * m_wheelSlipTractionMultiplier;
    else
      m_bWheelSlipping = false;
  }

  m_statAppliedForce = m_statEngineForce - m_statWheelslipForce;

  // Reduce traction when the wheels are slipping (avoid this at <1m/s to avoid
  // cases where the train is impossible to get moving).
  if (m_bWheelSlipping && fabs(wasVel) > 1.0)
    pushTorque *= m_wheelSlipTractionMultiplier;


  if (!m_bWheelSlipping)
  {
    // Add on any slope forces. It's ok if this overcomes the current velocity
    // and sends the wheel travelling in the opposite direction.
    double slopeForce = CalculateSlopeForce();
    m_statResistanceForce += slopeForce;
    pushTorque += slopeForce * wheelRadius * dt;
  }

  // Calculate the new wheel speed.
  m_angVel += pushTorque * dt / wheelRadius / wheelRadius / m_vehicleMass * 2;
  NANCheck(m_angVel);


  if (!m_bWheelSlipping)
  {
    // Apply track and internal resistance forces to the adjusted wheel speed,
    // making sure that the resistance can't flip direction.
    double groundResist = CalculateResistanceForces(GetWheelVelocity());
    m_statResistanceForce += groundResist;
    double decel = (groundResist * wheelRadius * dt * dt) / pow(wheelRadius, 2) / m_vehicleMass * 2;
    if (decel > fabs(m_angVel))
      m_angVel = 0;
    else
      m_angVel += decel * (m_angVel < 0 ? 1 : -1);
  }

  // Apply any braking forces to the wheels.
  ApplyBrakeFriction(dt);


  // Update the forward momentum for the new wheel velocity.
  if (!m_bWheelSlipping)
    m_forwardMomentum += (GetWheelVelocity() - wasVel) * m_vehicleMass;
  else
    m_forwardMomentum += (GetWheelVelocity() - wasVel) * m_vehicleMass * m_wheelSlipMomentumMultiplier;


  if (m_bWheelSlipping && m_angVel != 0)
  {
    // Now that we've applied the final (unslipped) wheel movement to momentum,
    // add the slipping portion straight onto the wheel. At this point we want
    // to affect the visuals of the wheel, not so much the physics.
    //double temp = m_angVel;

    double limit = (m_forwardMomentum * 2 / m_specWheelDiameter / m_vehicleMass) - m_angVel;
    m_angVel += Sign(limit) * std::min(fabs(limit), m_sandingFactor * 0.6 * m_vehicleMass * dt * wheelRadius / pow(wheelRadius, 2) / m_vehicleMass * 2);
    NANCheck(m_angVel);

    //if (temp / m_angVel < 0)
    //  m_angVel = 0;
  }


  // When the wheels are slipping we apply the resistance forces to the
  // train, not the wheels. This takes place as a final momentum adjustment.
  if (m_bWheelSlipping)
  {
    // Add on any slope forces. It's ok if this overcomes the forward momentum
    // and sends the train travelling in the opposite direction.
    double slopeForce = CalculateSlopeForce();
    m_statResistanceForce += slopeForce;
    m_forwardMomentum += slopeForce * dt;

    // Apply track and internal resistance forces, making sure that the
    // resistance can't flip direction.
    double groundResist = CalculateResistanceForces(GetTrainVelocity());
    m_statResistanceForce += groundResist;
    {
      double momentumChange = groundResist * dt;
      if (momentumChange > fabs(m_forwardMomentum))
        m_forwardMomentum = 0;
      else
        m_forwardMomentum += momentumChange * (m_forwardMomentum < 0 ? 1 : -1);
    }
  }


  // TT#2472 "Driver - Steam engines can reach impossible speeds."
  // The "maximum speed hack" clamps the output of the steam system such that
  // engine power cuts out after a pre-specified maximum velocity is exceeded.
  if (m_specMaximumSpeedHack > 0)
  {
    double speed = fabs(GetWheelVelocity());
    double prevSpeed = fabs(wasVel);

    if (speed > m_specMaximumSpeedHack && speed > prevSpeed)
      m_angVel = Sign(m_angVel) * m_specMaximumSpeedHack * 2 / m_specWheelDiameter;
  }

}


//=============================================================================
// Name: UpdatePosition
// Desc: Updates the position of the wheel (and pistons) over the time interval
//       passed, based on current angular velocity.
// Parm: dt - Time period to update the position for, in seconds.
//=============================================================================
void DrivingWheel::UpdatePosition(float dt)
{
  m_position += m_angVel * dt;
  while (m_position < 0)
    m_position += 2 * PI;

  m_position = fmod(m_position, 2 * PI);

  // With wheel movement comes piston movement (they are attached, after all).
  for (int i = 0; i < m_pistonCount; ++i)
    m_pistons[i].m_pist->SetPositionAbs(fmod(m_position + m_pistons[i].offsetOnWheel, 2 * PI));
}


//=============================================================================
// Name: AdjustMomentum
// Desc: Adjusts the momentum of the traincar. This is called to adjust the
//       traincar momentum after coupling forces are calculated.
// Parm: newMomentum - The new momentum of this traincar, in kg.m/s.
//=============================================================================
void DrivingWheel::AdjustMomentum(double newMomentum)
{
  m_forwardMomentum = newMomentum;

  // TODO: This isn't great, since a single frame pause in wheelslip will kick
  // this in and immediately wipe all wheel speed, ruining the visuals.
  if (!m_bWheelSlipping)
    m_angVel = 2 * (newMomentum / m_vehicleMass) / m_specWheelDiameter;
}


//=============================================================================
// Name: UpdateMass
// Desc: Updates the mass of this vehicle, typically as a product queue changes
//       (e.g. for fuel consumption/transfer).
// Parm: newVehicleMass - The new mass of this traincar and it's load, in kg.
//=============================================================================
void DrivingWheel::UpdateMass(double newVehicleMass)
{
  ASSERT(newVehicleMass > 0);

  // TODO: Check wheelslip? Adjust momentum? This approach seems quite strange.
  m_angVel *= m_vehicleMass / newVehicleMass;
  NANCheck(m_angVel);

  m_vehicleMass = newVehicleMass;
}


//=============================================================================
// Name: SetResistanceInfo
// Desc: Updates a few frame/environment-specific variables relevant to davis
//       formula resistance calcs.
// Parm: grade - The current (forward facing) track gradient, -1 to 1.
// Parm: trackCurvature - The current track curve angle, 0 to 180 degrees.
// Parm: weightTE - The tractive effort (Newtons) caused by vehicle gravity,
//       see TNIPhysicsVehicleStateSteam::DetermineTractiveEffortFromWeight().
// Parm: wheelslipTractionModifier - The amount to scale vehicle traction by
//       when under wheelslip (may be modified by TrainzScript).
// Parm: wheelslipMomentumModifier - The amount to scale vehicle momentum by
//       when under wheelslip (may be modified by TrainzScript).
//=============================================================================
void DrivingWheel::SetResistanceInfo(double grade, double trackCurvature, double weightTE,
                                     double wheelslipTractionModifier, double wheelslipMomentumModifier)
{
  m_currentGrade = grade;
  m_trackCurvature = trackCurvature;
  m_weightTE = weightTE;
  m_wheelSlipTractionMultiplier = wheelslipTractionModifier;
  m_wheelSlipMomentumMultiplier = wheelslipMomentumModifier;
}


//=============================================================================
// Name: ApplyBrakeFriction
// Desc: Applies the braking forces to wheel velocity.
// Parm: dt - Time period to update the brakes for, in seconds.
//=============================================================================
void DrivingWheel::ApplyBrakeFriction(float dt)
{
  // Do nothing if neither brake is applied.
  if (m_handBrakeSetting == 0 && m_brakePressurePSI == 0)
    return;

  // Determine the friction caused by the hand brake.
  double handbrakeFriction = m_handBrakeSetting * m_specHandBrakeMaxForce;

  // Determine the friction caused by normal brake.
  double normBrakeFriction = m_specBrakeFriction * m_brakePressurePSI * 4.448;

  m_statBrakeForce = handbrakeFriction + normBrakeFriction;

  if (m_angVel == 0 || dt == 0)
    return;

  const double wheelRadius = m_specWheelDiameter / 2;
  const double wasVel = m_angVel;

  // Combine the two, and convert that to a velocity loss for this frame.
  double velocityLoss = (-Sign(m_angVel) * (handbrakeFriction + normBrakeFriction) * wheelRadius * dt) /
                        pow(wheelRadius, 2) / m_vehicleMass * 2;

  m_angVel += velocityLoss;

  // Ensure the brakes only stop the wheel, and don't cause it to reverse direction.
  if (wasVel / m_angVel < 0)
    m_angVel = 0;
}


//=============================================================================
// Name: CalculateSlopeForce
// Desc: Returns the force of gravity on the traincar, based on track gradient.
//       This is added to the train engine force during update.
// Retn: double - The slope force on this traincar, in Newtons.
//=============================================================================
double DrivingWheel::CalculateSlopeForce(void)
{
  return m_currentGrade * m_vehicleMass * -9.8;
}


//=============================================================================
// Name: CalculateResistanceForces
// Desc: Calculates and returns the combined movement resistance forces using
//       the davis formula. This includes internal engine resistances, bogeys,
//       and track resistance (based on curvature). It does not include brake
//       forces, or slope/gradient force, which is applied separately.
// Parm: velocity - The current traincar velocity, in m/s
// Retn: double - The resitance force on this traincar, in Newtons.
//=============================================================================
double DrivingWheel::CalculateResistanceForces(double velocity)
{
  // Apply davis formula:
  //  R = (1.3 + 29.0 / w) + 0.045 * V + (0.0005 * a / w * n) * v * v
  //
  // Where:
  //  R = resistance (lbs / ton)
  //  w = unit weight per axle
  //  n = number of axles per car
  //  A = effective cross sectional area of car (sq. f)
  //  V = speed (miles / hr)
  //

  double weightInTons = UnitConversion::kg_ton(m_vehicleMass);
  double dv_n = m_specAxlecount;
  double dv_a = m_specSurfaceArea;
  double dv_w = weightInTons / dv_n;
  double dv_v = UnitConversion::mps_mph(abs(velocity));

  // Determine ground resistance as per davis-formula
  double resistance = 1.3 + 29.0 / dv_w + m_specMovingFrictionCoefficient * dv_v + ((m_specAirDragCoefficient * dv_a) / (dv_w * dv_n)) * dv_v * dv_v;

  // Add curve resistance
  resistance += (dv_v > UnitConversion::mph_mps(2.0) ? 0.8 : 1.0) * m_trackCurvature;

  // Multiply by weight to get resistance in pounds force
  resistance *= weightInTons;

  // And finally, convert to correct (SI) units (i.e. Newtons)
  resistance = UnitConversion::lbf_kn(resistance) * 1000;

  return resistance;
}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the wheel to the soup passed. Save will
//       always use the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the wheel state into. A sub-soup will be
//       created for this purpose, to limit the chance of name overlap.
//=============================================================================
void DrivingWheel::Save(TNIRef<TNISoup>& io_data) const
{
  TNIRef<TNISoup> wheelData = TNIAllocSoup();
  SoupSetFloat(wheelData, g_strings->lblTagPosition, m_position);
  SoupSetFloat(wheelData, g_strings->lblTagVelocity, m_angVel);
  SoupSetFloat(wheelData, g_strings->lblTagMomentum, m_forwardMomentum);
  SoupSetFloat(wheelData, g_strings->lblTagWheelSlip, m_bWheelSlipping ? 1.0 : 0.0);

  TNISetSoupKeyValue(io_data, g_strings->lblTagWheel, wheelData);

  // Everything else is either from the spec, or calculated in Update().
}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the wheel from the soup passed.
// Parm: data - The (read-only) soup to restore the wheel state from. Note
//       that this soup may not have been created by this plugin.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void DrivingWheel::Load(const TNIRef<const TNISoup>& data, int dataVersion)
{
  const TNIRef<const TNISoup> wheelData = TNICastSoup(TNIGetSoupValueByKey(data, g_strings->lblTagWheel));
  if (!wheelData)
    return;

  m_position = SoupGetFloat(wheelData, g_strings->lblTagPosition, m_position);
  m_angVel = SoupGetFloat(wheelData, g_strings->lblTagVelocity, m_angVel);
  m_forwardMomentum = SoupGetFloat(wheelData, g_strings->lblTagMomentum, m_forwardMomentum);
  m_bWheelSlipping = SoupGetFloat(wheelData, g_strings->lblTagWheelSlip, m_bWheelSlipping ? 1.0 : 0.0) > 0;


  // Sanity check the ranges on these variables.
  m_position = SanityCheckClamp(0, m_position, 2 * PI);

  NANCheck(m_position);
  NANCheck(m_angVel);
  NANCheck(m_forwardMomentum);

}


}; // namespace SteamPhysics

