//=============================================================================
// File: SteamPhysics_DrivingWheel.h
// Desc: Declares SteamPhysics::DrivingWheel, an object representing the
//       driving wheels within a steam engine.
//=============================================================================
#ifndef _SteamPhysics_DrivingWheel_H_
#define _SteamPhysics_DrivingWheel_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{

class Piston;


//=============================================================================
// Name: DrivingWheel
// Desc: The steam engine driving wheel. This takes force from the pistons and
//       transfers it into angular velocity, which is used to applu forward
//       momentum (and/or wheelslip).
//=============================================================================
class DrivingWheel
{
public:

  DrivingWheel( double vehicleMass, double diameter, double drivingWheelWeightRatio,
                double brakeRatio, double handbrakeMaxForce);

  void AddPiston(Piston* pis, double wheelOffset);
  void InitDavisFormulaValues(int axleCount, double surfaceArea, double movingFrictionCoefficient, double airDragCoefficient);
  void SetMaximumSpeedHack(double maximumSpeedHack) { m_specMaximumSpeedHack = maximumSpeedHack; }


  void Update(float dt);
  void UpdatePosition(float dt);

  void AdjustMomentum(double newMomentum);
  void UpdateMass(double newVehicleMass);


  void SetResistanceInfo(double grade, double trackCurvature, double weightTE,
                         double wheelslipTractionModifier, double wheelslipMomentumModifier);

  void SetHandBrakeSetting(double value) { m_handBrakeSetting = value; }
  void SetSandingFactor(double value) { m_sandingFactor = value; }
  void SetBrakePressurePSI(double value) { m_brakePressurePSI = value; }


  double GetForwardMomentum(void) const { return m_forwardMomentum; }
  double GetTrainVelocity(void) const { return m_forwardMomentum / m_vehicleMass; }

  void SetWheelRotation(double rotation) { m_position = rotation; }
  double GetWheelRotation(void) const { return m_position; }
  double GetAngularVelocity(void) const { return m_angVel; }
  double GetAngularMomentum(void) const { return m_angVel / 2 * m_vehicleMass * pow(m_specWheelDiameter / 2, 2); }
  double GetWheelVelocity(void) const { return m_angVel * m_specWheelDiameter / 2; }
  double GetWheelDiameter(void) const { return m_specWheelDiameter; }
  double GetEngineForce(void) const { return m_statEngineForce; }
  double GetAppliedEngineForce(void) const { return m_statAppliedForce; }
  double GetWheelslipForce(void) const { return m_statWheelslipForce; }
  bool GetWheelslip(void) const { return m_bWheelSlipping; }
  void SetHasWheelslip(bool bHasWheelslip) { m_bWheelSlipping = bHasWheelslip; }

  double GetBrakingForce(void) const { return m_statBrakeForce; }
  double GetResistanceForce(void) const { return m_statResistanceForce; }

  void Save(TNIRef<TNISoup>& io_data) const;
  void Load(const TNIRef<const TNISoup>& data, int dataVersion);



private:

  void ApplyBrakeFriction(float dt);

  double CalculateSlopeForce(void);
  double CalculateResistanceForces(double velocity);


  //===========================================================================
  class PistonInfo
  {
  public:
    PistonInfo(void) : m_pist(nullptr), offsetOnWheel(0) { }
    PistonInfo(Piston* p, double o) : m_pist(p), offsetOnWheel(o) { }

    Piston*     m_pist;         // The piston itself (not owned here).
    double      offsetOnWheel;  // The angular offset of the piston on the wheel.

  };


  // The pistons referenced by this object (not owned here)
  PistonInfo      m_pistons[STEAMPHYSICS_MAX_PISTONS];
  int             m_pistonCount;


  // These values are tracked and updated between frames, and need to be saved/loaded.
  double  m_position;           // The wheel rotation (0 to 2PI)
  double  m_angVel;             // The angular velocity of the wheel (m/s)
  double  m_forwardMomentum;    // The forward momentum of the vehicle (kg.m/s)
  bool    m_bWheelSlipping;     // Whether the wheels are currently slipping

  // These values are calculated each frame for stats/ui output only, and don't need to be saved
  double  m_statEngineForce;    // The amount of force from the engine this frame
  double  m_statAppliedForce;   // The amount of force applied this frame
  double  m_statWheelslipForce; // The amount of force lost this frame due to wheelslip
  double  m_statBrakeForce;     // The amount of resistance force from the brakes
  double  m_statResistanceForce;// The amount of resistance force from track, slope, etc

  // These values are set every frame by the calling code, and don't need to be saved
  double  m_sandingFactor;      // Current sanding setting, 0 to 1
  double  m_handBrakeSetting;   // Current handbrake setting, 0 to 1
  double  m_brakePressurePSI;   // Current brake cylinder pressure, in PSI

  double  m_vehicleMass;        // The current mass of the vehicle, in kg
  double  m_currentGrade;       // Track gradient, -1.0 to 1.0
  double  m_trackCurvature;     // Track curve angle, 0 to 180 degrees
  double  m_weightTE;           // Tractive effort caused by vehicle gravity, Newtons
  double  m_wheelSlipTractionMultiplier; // Traction scalar when under wheelslip
  double  m_wheelSlipMomentumMultiplier; // Momentum scalar when under wheelslip


  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.
  double  m_specWheelDiameter;      // Diameter of the wheels, in metres
  double  m_specDrivingWheelWeightRatio; // Ratio of vehicle mass on the driving wheels
  double  m_specBrakeFriction;      // Friction from the main brakes when active
  double  m_specHandBrakeMaxForce;  // Handbrake force when fully active, in Newtons
  double  m_specMaximumSpeedHack;   // Max speed override hack, used on certain content

  int     m_specAxlecount;          // Number of axles on this traincar (for resistance calc)
  double  m_specSurfaceArea;        // Cross-sectional surface area of the loco, in square feet
  double  m_specMovingFrictionCoefficient; // Friction coefficient which scales against speed
  double  m_specAirDragCoefficient; // Friction coefficient which scales against speed squared


};


}; // namespace SteamPhysics


#endif // _SteamPhysics_DrivingWheel_H_

