//=============================================================================
// File: SteamPhysics_FireBox.cpp
// Desc: Defines SteamPhysics::FireBox, an object representing the firebox
//       within a steam engine.
//=============================================================================
#include "SteamPhysics_FireBox.h"
#include "TNIFunctions.hpp"


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{


//=============================================================================
// Name: FireBox
// Desc: Steam fire box constructor. This takes a number of required values
//       which specify how the firebox behaves (all pulled from the engine
//       asset spec), and sets everything else to the defaults. Accessors are
//       provided to alter other spec values, see the class declaration.
// Parm: idleburnRate - The rate at which the firebox constantly/idly burns
//       coal, in kg/s (before being scaled down by drafting rate).
// Parm: idleTemperature - The idle temperature of the firebox, in Kelvin.
//       This is effectively our minimum fire temperature.
// Parm: burnRate - The tested/'normal' rate at which the firebox burns coal,
//       in kg/s (before being scaled down by drafting rate).
// Parm: testTemperature - The maximum fire temperature, in Kelvin.
// Parm: speed - The test speed value, in m/s. Used to determine
//       draft factor (see FireBox::Update() for more info).
// Parm: cutoff - The test cutoff value, (absoute) from 0 to 0.75. Used to
//       determine draft factor (see FireBox::Update() for more info).
// Parm: idealCoalMass - The ideal mass of coal to have in the firebox, in kg.
//       Any amount of coal above or below this will decrease efficiency.
// Parm: steamChestPressure - The test steam chest pressure, in KiloPascals.
//       Used to determine draft factor (see FireBox::Update() for more info).
//=============================================================================
FireBox::FireBox(double idleBurnRate, double idleTemperature, double burnRate,
                 double testTemperature, double speed, double cutoff,
                 double idealCoalMass, double steamChestPressure)
{
  m_temperature = (testTemperature + idleTemperature) / 2;
  m_energy = 0;
  m_draftFactor = 0;
  m_burnRateFactor = 0;
  m_blowerSetting = 0;

  m_specBurnRateTest = burnRate;
  m_specTemperatureTest = testTemperature;
  m_specTestSpeed = speed;
  m_specTestCutoff = cutoff;
  m_specIdealCoalMass = idealCoalMass;
  m_specTestSteamChestPressure = steamChestPressure * 1000;

  m_specBurnRateIdle = idleBurnRate;
  m_specTemperatureIdle = idleTemperature;

  SetupOtherParameters(0.5, 5, 0.025, 401, 27912000, 1260, 1000, 500);
  SetupBlowerParameters(0.5, 2.5);
}


//=============================================================================
// Name: SetupOtherParameters
// Desc: Sets various 'other' parameters defining firebox behaviour, as pulled
//       from the engine asset spec. Called during firebox construction.
// Parm: fireBoxEfficiency - The efficiency scalar for the firebox. This sets
//       the amount of energy that is lost by the firebox when burning coal.
// Parm: heatingSurfaceArea - The surface area of the firebox wall against the
//       boiler, in m^2. Used to calculate energy transfer into the boiler.
// Parm: fireboxThickness - The thickness of the firebox wall against the
//       boiler, in metres. Used to calculate energy transfer into the boiler.
// Parm: fireboxThermalConductivity - The thermal conductivity of the firebox
//       wall, in Watts per metre Kelvin W/(mK). Used to calculate energy
//       transfer into the boiler.
// Parm: fuelEnergy - The energy gained from burning the particular coal/fuel
//       that this firebox is using, in Joules per kilogram.
// Parm: fuelSpecificHeatCapacity - The specific heat capacity of the
//       particular coal/fuel that this firebox is using, in Joules per
//       Kilogram Kelvin. (i.e. The amount of energy required to raise the
//       temperature of the fuel by 1 degree Kelvin/Celcius.)
// Parm: maxCoal - The maximum mass of coal the firebox can hold, in kilograms.
// Parm: initialFuel - The initial mass of coal in the firebox, in kilograms.
//=============================================================================
void FireBox::SetupOtherParameters(double fireBoxEfficency, double heatingSurfaceArea,
                                   double fireboxThickness, double fireboxThermalConductivity,
                                   double fuelEnergy, double fuelSpecificHeatCapacity,
                                   double maxCoal, double initialFuel)
{
  m_specFireboxEfficency = fireBoxEfficency;
  m_specHeatingSurfaceArea = heatingSurfaceArea;
  m_specFireboxThickness = fireboxThickness;
  m_specFireboxThermalConductivity = fireboxThermalConductivity;
  m_specFuelEnergy = fuelEnergy;
  m_specFuelSpecificHeatCapacity = fuelSpecificHeatCapacity;
  m_specMaxCoal = maxCoal;

  m_fuelAmount = initialFuel;
}


//=============================================================================
// Name: SetupBlowerSettings
// Desc: Sets the blower specifications, as pulled from the engine asset spec.
//       Called during firebox construction.
// Parm: blowerEffect - The maximum effect of the blower on firebox draft.
// Parm: blowerMaxFlow - The flow rate of steam when draft is at maximum. See
//       GetBlowerSteamDemand() and Boiler::Update() for more info.
//=============================================================================
void FireBox::SetupBlowerParameters(double blowerEffect, double blowerMaxFlow)
{
  m_specBlowerEffect = blowerEffect;
  m_specBlowerMaxFlow = blowerMaxFlow;
}


//=============================================================================
// Name: Update
// Desc: Firebox update function. This takes the current speed and cutoff as
//       input parameters, as well as some boiler/steam-chest state, and
//       updates the firebox for the given time interval.
// Parm: dt - The time period to update the firebox for, in seconds.
// Parm: curSpeed - The current traincar speed (in m/s).
// Parm: curCutoff - The current absolute cutoff setting (0 to 0.75).
// Parm: curSteamChestPressure - The current steam chest pressure, in Pascals.
// Parm: curBoilerPressure - The current boiler pressure, in Pascals.
// Parm: curBoilerTemperature -The current boiler temperature, in Kelvin.
//=============================================================================
void FireBox::Update(float dt, double curSpeed, double curCutoff,
                     double curSteamChestPressure, double curBoilerPressure, double curBoilerTemperature)
{
  // Burn rate is proportional to draft, fire temperature, and coal mass.
  // Draft is proportional to steam chest pressure * cutoff setting (i.e.
  // exhaust beat size), and speed (i.e. number of exhaust beats).

  // To control performance, we have idle and test numbers. The test specifies
  // a cutoff, speed, steam chest pressure, fire temperature, and coal mass. The
  // plan is that ideal numbers for the loco should be provided for pressure,
  // temperature and mass.

  // The draft factor is calculated by linear interpolation between idle and
  // the test values. Cutoff and speed are linear or 1/2 speed = 1/2 fire rate,
  // 1/2 cutoff = 1/2 fire rate, 1/2 speed and 1/2 cutoff = 1/4 fire rate.

  m_draftFactor = (fabs(curSpeed) / m_specTestSpeed) * (fabs(curCutoff) / m_specTestCutoff) * (curSteamChestPressure / m_specTestSteamChestPressure);
  NANCheck(m_draftFactor);

  // TODO: Add the effect of dampers in here.


  // Blower has less effect as main draft increases.
  m_draftFactor += (m_blowerSetting * m_specBlowerEffect * (curBoilerPressure / m_specTestSteamChestPressure)) / (1 + m_draftFactor * 3);
  NANCheck(m_draftFactor);

  // Destroy the fire if draft too high.
  if (m_draftFactor > 1.0)
  {
    // Reduce rate of fire destruction depending on how much the loco is being thrashed.
    m_fuelAmount *= 1 - (m_draftFactor - 1) * dt * 0.01;
  }

  // Calculate the burn rate based on the draft, temperature and coalmass

  // 22/11/12 - Removed the effect that the temperature difference has here.
  // This should stop people getting stuck with a low fire temperature and no
  // ability to drive the loco out of it. Temperature has an effect later on
  // anyway - it's not necessary here.
  // 12/12/12 - Changed the 0..1 curve from a sqrt to a log based calculation.
  // This gives a closer result to the originally intended effect, allowing
  // more tolerance of "almost right" conditions. The 0.015 parameter has been
  // chosen to ensure that m_burnRateFactor never spikes negative (which it
  // otherwise could).
  m_burnRateFactor = 1.0 + (log(m_draftFactor + 0.015) / 4.0);
  NANCheck(m_burnRateFactor);


  if (m_fuelAmount > m_specIdealCoalMass)
  {
    // Reduce burn rate due to too much fuel in firebox.
    m_burnRateFactor /= (1 + (m_fuelAmount - m_specIdealCoalMass) / m_specIdealCoalMass);
    NANCheck(m_burnRateFactor);
  }
  else
  {
    // Reduce burn rate due to not enough fuel in firebox.
    m_burnRateFactor *= m_fuelAmount / m_specIdealCoalMass;
    NANCheck(m_burnRateFactor);
  }

  // Calculate the final burn rate.
  double burnRate = m_specBurnRateIdle + (m_specBurnRateTest - m_specBurnRateIdle) * m_burnRateFactor;

  // Amount of mass burnt for this time interval.
  double massBurnt = burnRate * dt;
  if (massBurnt > m_fuelAmount)
    massBurnt = m_fuelAmount;

  // Subtract the burnt mass from the fuel amount remaining.
  m_fuelAmount -= massBurnt;

  // Calculate the energy release from that mass.
  double energyReleased = massBurnt * m_specFuelEnergy * m_specFireboxEfficency;

  // Heat the fire with that energy.
  if (energyReleased > 0)
  {
    double heat = m_fuelAmount * m_specFuelSpecificHeatCapacity;
    if (heat > 0)
    {
      m_temperature += energyReleased / heat;
      NANCheck(m_temperature);
    }
  }

  // Cap temperature at maximum
  if (m_temperature > m_specTemperatureTest)
    m_temperature = m_specTemperatureTest;


  // Now what we have is a hotter fire. Time to work out how much heat transfer
  // we have through the firebox wall.
  //
  //  Delta Q           Delta T
  //  -------   =  -k A -------
  //  Delta t           Delta x
  //
  //  Delta Q is energy transfer (J)
  //  Delta t is timeslice (s)
  //  k is the thermal conductivity of the material. For us this is most likely copper, at 401 W/(m K).
  //  A is the heating surface area for the firebox (m^2)
  //  Delta T is the temperature difference between sides (K)
  //  Delta x is the thickness of the material (m)
  //
  // Formula for this comes from http://en.wikipedia.org/wiki/Heat_conduction

  // Heat transfer is determined by temperature gradient, however since we cap
  // the minimum temperature of the firebox, we would create an infinite heat
  // source when we should instead be draining all the coal and dropping to a
  // low energy state.
  // So, the below checks to make sure we have some realistic amount of coal
  // before allowing any heat transfer to occur. This will at least prevent
  // infinite heat transfer after we run out of coal.
  // We also ensure that we're actually generating heat; if the temperature is
  // below the idle temperature, then we are capped and not generating enough
  // heat to be worth transferring anything to the boiler.
  if (m_fuelAmount > m_specIdealCoalMass * 0.01f && m_temperature > m_specTemperatureIdle)
  {
    m_energy = m_specFireboxThermalConductivity * m_specHeatingSurfaceArea * ((m_temperature - curBoilerTemperature) / m_specFireboxThickness) * dt;
    NANCheck(m_energy);

    // Remove this energy from the fire temperature.
    m_temperature -= m_energy / (m_fuelAmount * m_specFuelSpecificHeatCapacity);
    NANCheck(m_temperature);

    // Cap temperature at minimum.
    if (m_temperature < m_specTemperatureIdle)
      m_temperature = m_specTemperatureIdle;
  }
  else
  {
    // We don't have enough coal left to give a serious temperature reading.
    m_temperature = m_specTemperatureIdle;
  }

}


//=============================================================================
// Name: UseEnergy
// Desc: Attempts to remove some amount of energy from the firebox, typically
//       to transfer it somewhere else (like the boiler, or perhaps into the
//       ambient environment).
// Parm: wantAmount - The amount of energy (Joules) the caller wants to remove.
// Retn: double - The amount of energy (Joules) that was actually removed. This
//       will differ from wantAmount if the caller requested more energy than
//       was available.
//=============================================================================
double FireBox::UseEnergy(float wantAmount)
{
  if (wantAmount > m_energy)
    wantAmount = (float)m_energy;

  m_energy -= wantAmount;
  return wantAmount;
}


//=============================================================================
// Name: UseAllEnergy
// Desc: Uses all energy in the firebox, returning the exact amount.
// Retn: double - The amount of energy (Joules) that was in the firebox, and
//       has now been removed.
//=============================================================================
double FireBox::UseAllEnergy(void)
{
  double wasEnergy = m_energy;
  m_energy = 0;
  return wasEnergy;
}


//=============================================================================
// Name: AddCoal
// Desc: Adds some amount of coal to the fire, either as an automated engine
//       process or in direct response to player input (e.g. the "shovel coal"
//       keyboard command).
// Parm: coalMass - The mass (kg) of coal to add.
//=============================================================================
void FireBox::AddCoal(double coalMass)
{
  // Average temperature between new coal (assume 273K) and existing fire temperature.
  // This means that coaling up quickly if you've let the fire burn down (or
  // thrashed it away) will reduce the fire temperature, and slow the burn rate.
  m_temperature = ((m_fuelAmount * m_temperature) + (coalMass * 273)) / (m_fuelAmount + coalMass);

  // Actually shovel the coal in.
  m_fuelAmount += coalMass;

  // Cap coal at max capacity (assume that any excess fell onto the track I guess).
  if (m_fuelAmount > m_specMaxCoal)
    m_fuelAmount = m_specMaxCoal;

  // Cap temperature at minimum, ignoring/avoiding the part where the fire
  // should actually go out, not because we can't do it, but because it's not
  // much fun for most players.
  if (m_temperature < m_specTemperatureIdle)
    m_temperature = m_specTemperatureIdle;
}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the firebox to the soup passed. Save will
//       always use the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the firebox state into. A sub-soup will be
//       created for this purpose, to limit the chance of name overlap.
//=============================================================================
void FireBox::Save(TNIRef<TNISoup>& io_data) const
{
  TNIRef<TNISoup> fireboxData = TNIAllocSoup();
  SoupSetFloat(fireboxData, g_strings->lblTagTemperature, m_temperature);
  SoupSetFloat(fireboxData, g_strings->lblTagFuel, m_fuelAmount);
  SoupSetFloat(fireboxData, g_strings->lblTagEnergy, m_energy);

  TNISetSoupKeyValue(io_data, g_strings->lblTagFirebox, fireboxData);

  // Everything else is either from the spec (which the game will save and
  // restore for us), or calculated in Update().
}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the firebox from the soup passed.
// Parm: data - The (read-only) soup to restore the firebox state from. Note
//       that this soup may not have been created by this plugin.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void FireBox::Load(const TNIRef<const TNISoup>& data, int dataVersion)
{
  const TNIRef<const TNISoup> fireboxData = TNICastSoup(TNIGetSoupValueByKey(data, g_strings->lblTagFirebox));
  if (!fireboxData)
    return;

  m_temperature = SoupGetFloat(fireboxData, g_strings->lblTagTemperature, m_temperature);
  m_fuelAmount = SoupGetFloat(fireboxData, g_strings->lblTagFuel, m_fuelAmount);
  m_energy = SoupGetFloat(fireboxData, g_strings->lblTagEnergy, m_energy);


  // Sanity check the ranges on these variables.
  m_temperature = SanityCheckClamp(m_specTemperatureIdle, m_temperature, m_specTemperatureTest);
  m_fuelAmount = SanityCheckClamp(0, m_fuelAmount, m_specMaxCoal);
  m_energy = SanityCheckClamp(0, m_energy, m_energy);
}


}; // namespace SteamPhysics

