//=============================================================================
// File: SteamPhysics_FireBox.h
// Desc: Declares SteamPhysics::FireBox, an object representing the firebox
//       within a steam engine.
//=============================================================================
#ifndef _SteamPhysics_FireBox_H_
#define _SteamPhysics_FireBox_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{


//=============================================================================
// Name: FireBox
// Desc: The steam engine firebox. This burns fuel to create heat energy, which
//       is then transferred to the boiler.
//=============================================================================
class FireBox
{
public:

  FireBox(double idleBurnRate, double idleTemperature, double burnRate, double temperature,
          double speed, double cutoff, double idealCoalMass, double steamChestPressure);

  void SetupOtherParameters(double efficency, double heatingSurfaceArea, double fireboxThickness, double thermalConductivity,
                            double fuelEnergy, double fuelSpecificHeatCapacity, double maxCoal, double currentFuel);
  void SetupBlowerParameters(double blowerEffect, double blowerMaxFlow);


  void Update(float dt, double curSpeed, double curCutoff, double curSteamChestPressure, double curBoilerPressure, double curBoilerTemperature);

  double UseEnergy(float amount);
  double UseAllEnergy(void);

  void AddCoal(double amount);
  double GetCoal(void) const { return m_fuelAmount; }
  double GetCurrentEnergy(void) const { return m_energy; }
  double GetPercentFull(void) const { return m_fuelAmount / m_specIdealCoalMass; }

  void SetBlowerSetting(float value) { m_blowerSetting = value; }
  double GetBlowerSetting(void) const { return m_blowerSetting; }

  double GetFireTemperature(void) const { return m_temperature; }
  double GetBlowerSteamDemand(void) const { return m_specBlowerEffect * m_specBlowerMaxFlow / 55.6; }

  double GetThrashingFactor(void) const { return m_draftFactor; }
  double GetBurnRateFactor(void) const { return m_burnRateFactor; }


  void Save(TNIRef<TNISoup>& io_data) const;
  void Load(const TNIRef<const TNISoup>& data, int dataVersion);


private:

  // These values are tracked and updated between frames, and need to be saved/loaded.
  double m_fuelAmount;            // kg of coal.
  double m_energy;                // J of energy generated to use.
  double m_temperature;           // Kelvin

  // These values are derived each frame, and don't need to be saved/loaded.
  double m_draftFactor;           // Current draft factor, 0 to 1
  double m_burnRateFactor;        // Burn rate factor (derived from draft)

  // The blower setting is updated every frame by the calling code.
  double m_blowerSetting;         // Current player blower setting

  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.

  double m_specBlowerEffect;      // Maximum effect of the blower on draft
  double m_specBlowerMaxFlow;     // Flow rate of steam when draft is at maximum
  double m_specBurnRateIdle;      // kg of coal/sec (at max draft)
  double m_specBurnRateTest;      // kg of coal/sec (at max draft)
  double m_specTemperatureIdle;   // Kelvin
  double m_specTemperatureTest;   // Kelvin
  double m_specMaxCoal;           // Max kg of coal
  double m_specIdealCoalMass;     // Ideal kg of coal
  double m_specFuelEnergy;        // J/kg
  double m_specFuelSpecificHeatCapacity; // J/kg K
  // Anthracite coal has somewhere between 26,000,000 and 33,000,000 J/kg of
  // energy, and a specific heat capacity of about 1260 J/kg K. Bituminous coal
  // has somewhere between 24,000,000 and 35,000,000 J/kg energy, and a
  // specific heat capacity of about 1380 J/kg K.

  // Drafting factor calculation 'test' values.
  double m_specTestSpeed;         // Traincar speed, in m/s
  double m_specTestCutoff;        // Absolute cutoff setting, 0 to 0.75
  double m_specTestSteamChestPressure; // Steam chest pressure, Pascals

  double m_specFireboxEfficency;  // Efficiency scalar, from 0 to 1;
  double m_specHeatingSurfaceArea;// Square metres (m^2)
  double m_specFireboxThermalConductivity; // W/(m K); copper is 401 W/(m K), steel varies depending on construction.
  double m_specFireboxThickness;  // m -- e.g. ~0.025 would be 1" thick plate

};


}; // namespace SteamPhysics


#endif // _SteamPhysics_FireBox_H_

