//=============================================================================
// File: SteamPhysics_Piston.cpp
// Desc: Defines SteamPhysics::Piston, an object representing a single piston
//       within a steam engine. See also, SteamPhysics_PistonManager.
//=============================================================================
#include "SteamPhysicsComponents.h"
#include "TNIFunctions.hpp"
#include "UnitConversion.h"
#include <algorithm>


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{


//=============================================================================
// Name: Piston
// Desc: Steam engine piston constructor. Takes a number of required values
//       which specify how the piston behaves (all pulled from the engine
//       asset spec), and sets everything else to the defaults.
// Parm: surfaceArea - The surface area of the piston face, in square metres
//       (m^2). This is not the surface area of the entire piston, just the
//       part which is exposed to the steam. Used to calculate piston force.
// Parm: minVol - The minimum volume of the piston cylinder (ie, the volume
//       when the piston is fully contracted), in Litres.
// Parm: minVol - The maximum volume of the piston cylinder (ie, the volume
//       when the piston is fully extended), in Litres.
// Parm: pos - The starting piston position, from -1 to 1.
//=============================================================================
Piston::Piston(double surfaceArea, double minVol, double maxVol, double pos)
{
  m_specPistonSurfaceArea = surfaceArea;
  m_specPistonMinVolume = minVol;
  m_specPistonMaxVolume = maxVol;
  m_specPistonVolume = minVol + maxVol;
  m_specPistonStrokeLength = (maxVol - minVol) / surfaceArea;

  m_leftMoles = 0;
  m_leftTemp = 273;
  m_rightMoles = 0;
  m_rightTemp = 273;

  m_pistonPosition = pos;
  m_pistonPositionAbs = 0;
}


//=============================================================================
// Name: CalculateForce
// Desc: Calculates and returns the internal force, based on the pressure on
//       either side of the piston in the cylinder.
// Retn: double - The calculated piston force, in Newtons.
//=============================================================================
double Piston::CalculateForce(void)
{
  double leftPressure = GetLeftPressure();
  double rightPressure = GetRightPressure();

  return (leftPressure - rightPressure) * m_specPistonSurfaceArea;
}


//=============================================================================
// Name: GetLeftPressure
// Desc: Returns the pressure on the 'left' side of the piston.
// Parm: double - The pressure (pascals) on the 'left' side of the piston.
//=============================================================================
double Piston::GetLeftPressure(void)
{
  double pressure = m_leftMoles * RCONST * m_leftTemp / GetLeftVolume();
  NANCheck(pressure);

  if (pressure < kAtmosphericPressurePascals)
  {
    pressure = kAtmosphericPressurePascals;
    m_leftMoles = pressure * GetLeftVolume() / RCONST / m_leftTemp;
    NANCheck(m_leftMoles);
  }

  return pressure;
}


//=============================================================================
// Name: GetRightPressure
// Desc: Returns the pressure on the 'right' side of the piston.
// Retn: double - the pressure (pascals) on the 'right' side of the piston.
//=============================================================================
double Piston::GetRightPressure(void)
{
  double pressure = m_rightMoles * RCONST * m_rightTemp / GetRightVolume();
  NANCheck(pressure);

  if (pressure < kAtmosphericPressurePascals)
  {
    pressure = kAtmosphericPressurePascals;
    m_rightMoles = pressure * GetRightVolume() / RCONST / m_rightTemp;
    NANCheck(m_rightMoles);
  }

  return pressure;
}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the piston to the soup passed. Save will
//       always use the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the piston state into. A sub-soup will be
//       created for this purpose, to limit the chance of name overlap.
// Parm: pistonIndex - The index of the piston being saved (used in the name
//       of the sub-soup).
//=============================================================================
void Piston::Save(TNIRef<TNISoup>& io_data, int pistonIndex) const
{
  ASSERT(pistonIndex >= 0 && pistonIndex < 10);

  TNIRef<TNISoup> pistonData = TNIAllocSoup();
  SoupSetFloat(pistonData, g_strings->lblTagPosition, m_pistonPositionAbs);
  SoupSetFloat(pistonData, g_strings->lblTagLeftMoles, m_leftMoles);
  SoupSetFloat(pistonData, g_strings->lblTagLeftTemp, m_leftTemp);
  SoupSetFloat(pistonData, g_strings->lblTagRightMoles, m_rightMoles);
  SoupSetFloat(pistonData, g_strings->lblTagRightTemp, m_rightTemp);

  char buffer[10];
  snprintf(buffer, 10, "piston%d", pistonIndex);
  TNIRef<TNILabel> soupLabel = TNIAllocLabel(buffer);

  TNISetSoupKeyValue(io_data, soupLabel, pistonData);

  // Everything else is from the spec, and doesn't need to be saved.

}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the piston from the soup passed.
// Parm: data - The (read-only) soup to restore the piston state from. Note
//       that this soup may not have been created by this plugin.
// Parm: pistonIndex - The index of the piston being saved (used in the name
//       of the sub-soup).
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void Piston::Load(const TNIRef<const TNISoup>& data, int pistonIndex, int dataVersion)
{
  char buffer[10];
  snprintf(buffer, 10, "piston%d", pistonIndex);
  TNIRef<TNILabel> soupLabel = TNIAllocLabel(buffer);

  const TNIRef<const TNISoup> pistonData = TNICastSoup(TNIGetSoupValueByKey(data, soupLabel));
  if (!pistonData)
    return;

  m_pistonPositionAbs = SoupGetFloat(pistonData, g_strings->lblTagPosition, m_pistonPositionAbs);
  m_leftMoles = SoupGetFloat(pistonData, g_strings->lblTagLeftMoles, m_leftMoles);
  m_leftTemp = SoupGetFloat(pistonData, g_strings->lblTagLeftTemp, m_leftTemp);
  m_rightMoles = SoupGetFloat(pistonData, g_strings->lblTagRightMoles, m_rightMoles);
  m_rightTemp = SoupGetFloat(pistonData, g_strings->lblTagRightTemp, m_rightTemp);


  // Sanity check the ranges on these variables.
  m_pistonPositionAbs = SanityCheckClamp(0, m_pistonPositionAbs, 2*PI);
  m_leftMoles = SanityCheckClamp(0, m_leftMoles, m_leftMoles);
  m_leftTemp = SanityCheckClamp(273, m_leftTemp, m_leftTemp);
  m_rightMoles = SanityCheckClamp(0, m_rightMoles, m_rightMoles);
  m_rightTemp = SanityCheckClamp(273, m_rightTemp, m_rightTemp);


  // Piston position can just be derived on load.
  m_pistonPosition = sin(m_pistonPositionAbs);

}


}; // namespace SteamPhysics

