//=============================================================================
// File: SteamPhysics_Piston.h
// Desc: Declares SteamPhysics::Piston, an object representing a single piston
//       within a steam engine. See also, SteamPhysics_PistonManager.
//=============================================================================
#ifndef _SteamPhysics_Piston_H_
#define _SteamPhysics_Piston_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{


//=============================================================================
// Name: Piston
// Desc: A steam engine piston. This is a cylinder containing a moving piston
//       head, with pressurised high temperature steam on either side. The
//       variance in pressure either side of the piston generates force, which
//       is later transferred to the wheels.
//       Control of steam into and out of the pistons is the responsibility of
//       the PistonManager, declared below.
//=============================================================================
class Piston
{
public:

  Piston(double surfaceArea, double minVol, double maxVol, double pos);

  double CalculateForce(void);

  double GetPistonPosition(void) const { return m_pistonPosition; }
  void SetPositionAbs(double pos) { m_pistonPositionAbs = pos; m_pistonPosition = sin(pos); }
  double GetPistonPositionAbs(void) const { return m_pistonPositionAbs; }

  double GetLeftPressure(void);
  double GetRightPressure(void);

  double GetLeftVolume(void) const { return m_specPistonMinVolume + ((m_pistonPosition + 1) / 2) * (m_specPistonMaxVolume - m_specPistonMinVolume); }
  double GetRightVolume(void) const { return m_specPistonMinVolume + (1 - ((m_pistonPosition + 1) / 2)) * (m_specPistonMaxVolume - m_specPistonMinVolume); }

  void SetLeftSteam(double amount) { m_leftMoles = amount; }
  void AddLeftSteam(double amount) { m_leftMoles += amount; }
  double GetLeftSteam(void) const { return m_leftMoles; }

  void SetRightSteam(double amount) { m_rightMoles = amount; }
  void AddRightSteam(double amount) { m_rightMoles += amount; }
  double GetRightSteam(void) const { return m_rightMoles; }

  void SetLeftTemperature(double temp) { m_leftTemp = temp; }
  double GetLeftTemperature(void) const { return m_leftTemp; }
  void SetRightTemperature(double temp) { m_rightTemp = temp; }
  double GetRightTemperature(void) const { return m_rightTemp; }

  double GetPistonStrokeLength(void) { return m_specPistonStrokeLength; }

  void Save(TNIRef<TNISoup>& io_data, int pistonIndex) const;
  void Load(const TNIRef<const TNISoup>& data, int pistonIndex, int dataVersion);


private:


  // These values are tracked and updated between frames, and need to be saved/loaded.
  double m_pistonPosition;        // ranges from -1 to 1
  double m_pistonPositionAbs;     // ranges from 0 to 2PI
  double m_leftMoles;             // L
  double m_leftTemp;              // K (Kelvins)
  double m_rightMoles;            // L
  double m_rightTemp;             // K (Kelvins)


  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.
  double m_specPistonSurfaceArea; // Squared meters
  double m_specPistonVolume;      // L
  double m_specPistonMinVolume;   // L
  double m_specPistonMaxVolume;   // L
  double m_specPistonStrokeLength;// m

};


}; // namespace SteamPhysics


#endif // _SteamPhysics_Piston_H_

