//=============================================================================
// File: SteamPhysics_PistonManager.cpp
// Desc: Defines SteamPhysics::PistonManager, an object used to manage the
//       pistons (SteamPhysics::Piston) of a steam engine.
//=============================================================================
#include "SteamPhysicsComponents.h"
#include "TNIFunctions.hpp"
#include "UnitConversion.h"
#include <algorithm>


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{


//=============================================================================
// Name: PistonManager
// Desc: Steam engine piston constructor. Takes a number of required values
//       which specify how the pistons manager behaves (all pulled from the
//       engine asset spec), and sets everything else to the defaults.
//       By default, no pistons exist in the piston manager. They must be
//       separately constructed and initialised, and added using AddPiston().
// Parm: valveLapPercent - Portion on top of the engine cutoff setting where
//       steam is allowed to escape the pistons into the atmosphere.
//=============================================================================
PistonManager::PistonManager(double valveLapPercent)
{
  for (int i = 0; i < STEAMPHYSICS_MAX_PISTONS; ++i)
    m_pistons[i] = nullptr;
  m_pistonCount = 0;

  m_specValveLapPercent = valveLapPercent;
}


//=============================================================================
// Name: AddPiston
// Desc: Adds the piston passed to the internal array. The piston manager
//       references the pistons only, and does not own/delete them.
// Parm: piston - The pison to add to this piston manager. The caller retains
//       ownership of this object, and must delete it themselves (ideally after
//       the piston manager, or at least after the piston manager is last used.
//=============================================================================
void PistonManager::AddPiston(Piston* piston)
{
  if (m_pistonCount >= STEAMPHYSICS_MAX_PISTONS)
  {
    ASSERT(false);
    return;
  }

  m_pistons[m_pistonCount] = piston;
  ++m_pistonCount;
}


//=============================================================================
// Name: Update
// Desc: Updates the pistons, letting steam into them from the steam chest,
//       or out of them into the atmosphere, based on the piston position and
//       engine cutoff setting.
// Parm: steamChest - The steam chest
// Parm: dt - The time interval to update the pistons for, in seconds.
// Parm: cutoffSetting - The current player cutoff setting (-0.75 to 0.75).
//=============================================================================
void PistonManager::Update(SteamChest* steamChest, float dt, double cutoffSetting)
{
  #define FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS    101325.01 //135798  // = 5PSI above atmosphere in Pascals
  #define DIRECTION_FORWARD                       -1
  #define DIRECTION_NONE                          0
  #define DIRECTION_BACKWARD                      1

  for (int i = 0; i < m_pistonCount; ++i)
  {
    Piston* m_piston = m_pistons[i];
    double pistonPos = (m_piston->GetPistonPosition() + 1) / 2.0;
    int direction = m_piston->GetPistonPositionAbs() < 0.5 * PI || m_piston->GetPistonPositionAbs() > 1.5 * PI  ? DIRECTION_BACKWARD : DIRECTION_FORWARD;


    if (cutoffSetting > 0)
    {
      // Positive cutoff, i.e. forward gear

      direction = -direction;
      if (direction == DIRECTION_BACKWARD || direction == DIRECTION_NONE)
      {
        if (pistonPos > 1 - cutoffSetting - m_specValveLapPercent)
        {
          if (pistonPos > 1 - cutoffSetting)
          {
            // Add steam to piston right (from steam chest)
            double c1 = steamChest->GetPressure() * steamChest->GetVolume() / steamChest->GetTemperature();
            NANCheck(c1);
            double c2 = m_piston->GetRightPressure() * m_piston->GetRightVolume() / m_piston->GetRightTemperature();
            NANCheck(c2);
            double amount = (c1 + c2) * steamChest->GetTemperature() / (m_piston->GetRightVolume() + steamChest->GetVolume());
            NANCheck(amount);

            double moles = amount * m_piston->GetRightVolume() / (RCONST * steamChest->GetTemperature());
            NANCheck(moles);
            if (moles - m_piston->GetRightSteam() > 0)
            {
              m_piston->SetRightTemperature(steamChest->GetTemperature());
              amount = steamChest->RemoveSteam(moles - m_piston->GetRightSteam());
              m_piston->AddRightSteam(amount);
            }
          }

          m_piston->SetLeftTemperature(300.0);
          double amount2 = FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
        else if (pistonPos < m_specValveLapPercent)
        {
          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetRightPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
        else
        {
          //m_piston->SetLeftTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
      }
      else if (direction == DIRECTION_FORWARD || direction == DIRECTION_NONE)
      {
        if (pistonPos < cutoffSetting + m_specValveLapPercent)
        {
          if (pistonPos < cutoffSetting)
          {
            // Add steam to piston left (from steam chest)
            double c1 = steamChest->GetPressure() * steamChest->GetVolume() / steamChest->GetTemperature();
            NANCheck(c1);
            double c2 = m_piston->GetLeftPressure() * m_piston->GetLeftVolume() / m_piston->GetLeftTemperature();
            NANCheck(c2);
            double amount = (c1 + c2) * steamChest->GetTemperature() / (m_piston->GetLeftVolume() + steamChest->GetVolume());
            NANCheck(amount);

            double moles = amount * m_piston->GetLeftVolume() / (RCONST * steamChest->GetTemperature());
            NANCheck(moles);
            if (moles - m_piston->GetLeftSteam() > 0)
            {
              m_piston->SetLeftTemperature(steamChest->GetTemperature());
              amount = steamChest->RemoveSteam(moles - m_piston->GetLeftSteam());
              m_piston->AddLeftSteam(amount);
            }
          }

          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
        else if (pistonPos > 1 - m_specValveLapPercent)
        {
          //m_piston->SetLeftTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
        else
        {
          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetRightPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
      }
    }
    else if (cutoffSetting < 0)
    {
      // Negative cutoff, i.e. reverse gear

      if (direction == DIRECTION_BACKWARD || direction == DIRECTION_NONE)
      {
        if (pistonPos > 1 + cutoffSetting - m_specValveLapPercent)
        {
          if (pistonPos > 1 + cutoffSetting)
          {
            // Add steam to piston right (from steam chest)
            double c1 = steamChest->GetPressure() * steamChest->GetVolume() / steamChest->GetTemperature();
            NANCheck(c1);
            double c2 = m_piston->GetRightPressure() * m_piston->GetRightVolume() / m_piston->GetRightTemperature();
            NANCheck(c2);
            double amount = (c1 + c2) * steamChest->GetTemperature() / (m_piston->GetRightVolume() + steamChest->GetVolume());
            NANCheck(amount);

            double moles = amount * m_piston->GetRightVolume() / (RCONST * steamChest->GetTemperature());
            NANCheck(moles);
            if (moles - m_piston->GetRightSteam() > 0)
            {
              m_piston->SetRightTemperature(steamChest->GetTemperature());
              amount = steamChest->RemoveSteam(moles - m_piston->GetRightSteam());
              m_piston->AddRightSteam(amount);
            }
          }

          //m_piston->SetLeftTemperature(300.0);
          double amount2 = (m_piston->GetRightPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
        else if (pistonPos < m_specValveLapPercent)
        {
          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetRightPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
        else
        {
          //m_piston->SetLeftTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
      }
      else if (direction == DIRECTION_FORWARD || direction == DIRECTION_NONE)
      {
        if (pistonPos < -cutoffSetting + m_specValveLapPercent)
        {
          if (pistonPos < -cutoffSetting)
          {
            // Add steam to piston left (from steam chest)
            double c1 = steamChest->GetPressure() * steamChest->GetVolume() / steamChest->GetTemperature();
            NANCheck(c1);
            double c2 = m_piston->GetLeftPressure() * m_piston->GetLeftVolume() / m_piston->GetLeftTemperature();
            NANCheck(c2);
            double amount = (c1 + c2) * steamChest->GetTemperature() / (m_piston->GetLeftVolume() + steamChest->GetVolume());
            NANCheck(amount);

            double moles = amount * m_piston->GetLeftVolume() / (RCONST * steamChest->GetTemperature());
            NANCheck(moles);
            if (moles - m_piston->GetLeftSteam() > 0)
            {
              m_piston->SetLeftTemperature(steamChest->GetTemperature());
              amount = steamChest->RemoveSteam(moles - m_piston->GetLeftSteam());
              m_piston->AddLeftSteam(amount);
            }
          }

          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
        else if (pistonPos > 1 - m_specValveLapPercent)
        {
          //m_piston->SetLeftTemperature(300.0);
          double amount2 = (m_piston->GetLeftPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetLeftVolume() / (RCONST * m_piston->GetLeftTemperature());
          NANCheck(moles);
          m_piston->SetLeftSteam(moles);
        }
        else
        {
          //m_piston->SetRightTemperature(300.0);
          double amount2 = (m_piston->GetRightPressure() + FIVE_PSI_ABOVE_ATMOSPHERE_IN_PASCALS) / 2;
          double moles = amount2 * m_piston->GetRightVolume() / (RCONST * m_piston->GetRightTemperature());
          NANCheck(moles);
          m_piston->SetRightSteam(moles);
        }
      }
    }
    else
    {
      // Cutoff exactly 0. Unhandled.
    }

  } // for (int i = 0; i < m_pistonCount; ++i)

}


}; // namespace SteamPhysics

