//=============================================================================
// File: SteamPhysics_SafetyValve.cpp
// Desc: Defines SteamPhysics::SafetyValve, an object representing a safety
//       valve on a steam engine boiler (SteamPhysics::Boiler).
//=============================================================================
#include "SteamPhysics_SafetyValve.h"
#include "SteamPhysics_Boiler.h"
#include "TNIFunctions.hpp"
#include <algorithm>


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{

//=============================================================================
// Name: SafetyValve
// Desc: Boiler safety valve constructor. This takes a number of required
//       values which specify how the valve behaves (all pulled from the engine
//       asset spec), and sets everything else to the defaults.
// Parm: releaseValue - The boiler pressure at which the valve should open and
//       release steam, in Pascals.
// Parm: flowRate - The rate at which the valve should release steam, in kg/s.
//=============================================================================
SafetyValve::SafetyValve(double releaseValue, double flowRate)
{
  m_specReleaseValue = std::max(releaseValue, 0.0);
  m_specCutoffValue = std::max(releaseValue - 34473.0, 0.0);
  m_specFlowRate = std::max(flowRate, 0.0);

  m_bIsTriggered = false;
}


//=============================================================================
// Name: Update
// Desc: Updates the safety valve over the time interval passed. First checking
//       if the valve should open or close, then releasing pressure if open.
// Parm: boiler - The boiler to check if we should release steam from.
// Parm: dt - Time period to update for, in seconds.
//=============================================================================
void SafetyValve::Update(Boiler* boiler, float dt)
{
  // TODO: We should really have Boiler inherit from something more generic,
  // and pass that through to here, so that we can use this class on any
  // pressurised vessel, but there's no usecase to justify it yet.
  double boilerPressure = boiler->GetBoilerPressure();

  // We enforce a minimum cutoff value here, to avoid configurations that
  // would break the engine.
  if (boilerPressure <= 101400.0)
    m_bIsTriggered = false;
  else if (boilerPressure > m_specReleaseValue)
    m_bIsTriggered = true;
  else if (m_bIsTriggered)
    m_bIsTriggered = boilerPressure > m_specCutoffValue;

  if (m_bIsTriggered)
    boiler->RemoveSteam(m_specFlowRate * dt);

}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the wheel to the soup passed. Save will
//       always use the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the valve state into. A sub-soup will be
//       created for this purpose, to limit the chance of name overlap.
// Parm: tagName - The unique name to use for the sub-soup.
//=============================================================================
void SafetyValve::Save(TNIRef<TNISoup>& io_data, const TNIRef<TNILabel>& tagName) const
{
  TNIRef<TNISoup> valveData = TNIAllocSoup();
  SoupSetInteger(valveData, g_strings->lblTagTriggered, m_bIsTriggered ? 1 : 0);

  TNISetSoupKeyValue(io_data, tagName, valveData);
}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the valve from the soup passed.
// Parm: data - The (read-only) soup to restore the valve state from. Note
//       that this soup may not have been created by this plugin.
// Parm: tagName - The unique name of the sub-soup the data is saved in.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void SafetyValve::Load(const TNIRef<const TNISoup>& data, const TNIRef<TNILabel>& tagName, int dataVersion)
{
  // We only have one value, but still use a sub-soup for ease of future extension
  const TNIRef<const TNISoup> valveData = TNICastSoup(TNIGetSoupValueByKey(data, tagName));
  if (!valveData)
    return;

  m_bIsTriggered = SoupGetInteger(valveData, g_strings->lblTagTriggered, m_bIsTriggered ? 1 : 0) > 0;
}


}; // namespace SteamPhysics

