//=============================================================================
// File: SteamPhysics_SafetyValve.h
// Desc: Declares SteamPhysics::SafetyValve, an object representing a safety
//       valve on a steam engine boiler (SteamPhysics::Boiler).
//=============================================================================
#ifndef _SteamPhysics_SafetyValve_H_
#define _SteamPhysics_SafetyValve_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{


//=============================================================================
// Name: SafetyValve
// Desc: A steam engine safety valve. While triggered, this releases steam from
//       the boiler into the atmosphere at a constant flow rate.
//=============================================================================
class SafetyValve
{
public:

  SafetyValve(double releaseValue, double flowRate);

  void Update(class Boiler* boiler, float dt);

  bool GetIsTriggered(void) const { return m_bIsTriggered; }
  double GetCurrentFlowRate(void) const { return m_bIsTriggered ? m_specFlowRate : 0; }

  void Save(TNIRef<TNISoup>& io_data, const TNIRef<TNILabel>& tagName) const;
  void Load(const TNIRef<const TNISoup>& data, const TNIRef<TNILabel>& tagName, int dataVersion);


private:

  // These values are tracked and updated between frames, and need to be saved/loaded.
  bool        m_bIsTriggered;     // Whether the valve is open

  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.
  double      m_specReleaseValue; // Pressure at which the valve releases
  double      m_specCutoffValue;  // Pressure at which the valve reengages
  double      m_specFlowRate;     // Rate at which pressure is released

};


}; // namespace SteamPhysics


#endif // _SteamPhysics_SafetyValve_H_

