//=============================================================================
// File: SteamPhysics_SteamChest.cpp
// Desc: Defines SteamPhysics::SteamChest, an object representing the steam
//       chest within a steam engine.
//=============================================================================
#include "SteamPhysicsComponents.h"
#include "TNIFunctions.hpp"
#include "UnitConversion.h"


extern TNIPhysicsStrings* g_strings;

namespace SteamPhysics
{


//=============================================================================
// Name: SteamChest
// Desc: Steam chest constructor. This takes a number of required values which
//       specify how the steam chest behaves (all pulled from the engine asset
//       spec), and sets anything else to the defaults.
// Parm: boiler - The boiler which we'll be transfering steam from (used for
//       copying some initial state).
// Parm: volume - The volume of the steam chest, in cubic metres (m^3).
// Parm: superheatingConstant - The temperature difference between saturated
//       (boiler) and superheated (steam chest) steam for this engine (in
//       Kelvin/Celcius).
// Parm: maxFlow - The maximum rate at which steam can flow from the boiler
//       into the steam chest (in moles per second).
//=============================================================================
SteamChest::SteamChest(Boiler* boiler, double volume, double superheatingConstant, double maxFlow)
{
  m_steamMoles = 0;
  m_steamTemp = boiler->GetBoilerWaterTemp() + superheatingConstant;

  m_regulatorSetting = 0.0;

  m_specVolume = volume;
  m_specMaxFlowRate = maxFlow;
  m_specSuperHeatingConstant = superheatingConstant;
}


//=============================================================================
// Name: Update
// Desc: Transfers steam from the boiler based on the regulator setting.
// Parm: boiler - The boiler to transfer steam from.
// Parm: dt - The time interval to update the steam chest for, in seconds.
//=============================================================================
void SteamChest::Update(Boiler* boiler, float dt)
{
  // TODO: Calculate local steam temperature based on incoming steam, rather
  // than just taking the boiler temperature?
  m_steamTemp = boiler->GetBoilerSteamTemp() + m_specSuperHeatingConstant;

  double c1 = GetPressure() * m_specVolume / m_steamTemp;
  NANCheck(c1);
  double c2 = boiler->GetBoilerPressure() * boiler->GetBoilerSteamVolume() / boiler->GetBoilerSteamTemp();
  NANCheck(c2);
  double amount = (c1 + c2) * boiler->GetBoilerSteamTemp() / (m_specVolume + boiler->GetBoilerSteamVolume());
  NANCheck(amount);
  double moles = amount * m_specVolume / (RCONST * m_steamTemp);
  NANCheck(moles);

  const double maxFlowThisFrame = m_specMaxFlowRate * m_regulatorSetting * dt;
  if (moles > m_steamMoles + maxFlowThisFrame)
    moles = m_steamMoles + maxFlowThisFrame;

  // Attempt to remove this steam from the boiler, allowing for the possibility
  // there isn't enough there.
  amount = boiler->RemoveSteam((moles - m_steamMoles) / 55.6);
  ASSERT(amount >= 0);
  m_steamMoles += amount * 55.6;
}


//=============================================================================
// Name: RemoveSteam
// Desc: Removes steam from the steam chest. This would typically be called
//       when transfering steam from the steam chest into the pistons.
// Parm: moles - The amount of steam (in moles) to remove.
// Retn: double - The amount of steam (in moles) that was actually removed.
//       This will differ from 'moles' if the caller requested more steam than
//       was actually available.
//=============================================================================
double SteamChest::RemoveSteam(double moles)
{
  ASSERT(moles >= 0);

  // Don't allow the caller to remove more than there is.
  if (moles > m_steamMoles)
    moles = m_steamMoles;

  m_steamMoles -= moles;
  return moles;
}


//=============================================================================
// Name: GetPressure
// Desc: Returns the internal pressure of the steam chest, in Pascals.
// Retn: double - The internal pressure (pascals) of the steam chest.
//=============================================================================
double SteamChest::GetPressure(void)
{
  double pressure = m_steamMoles * RCONST * m_steamTemp / m_specVolume;
  if (pressure < kAtmosphericPressurePascals)
  {
    m_steamMoles = kAtmosphericPressurePascals * m_specVolume / RCONST / m_steamTemp;
    pressure = kAtmosphericPressurePascals;
  }

  return pressure;
}


//=============================================================================
// Name: GetPressurePSI
// Desc: Returnes the internal pressure of the steam chest, in PSI.
// Retn: double - The internal pressure (PSI) of the steam chest.
//=============================================================================
double SteamChest::GetPressurePSI()
{
  return UnitConversion::pa_psi(GetPressure());
}


//=============================================================================
// Name: Save
// Desc: Saves the internal state of the steam chest to the soup passed. Always
//       uses the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - The soup to save the steam chest state into. A sub-soup will
//       be created for this purpose, to limit the chance of name overlap.
//=============================================================================
void SteamChest::Save(TNIRef<TNISoup>& io_data) const
{
  TNIRef<TNISoup> chestData = TNIAllocSoup();
  SoupSetFloat(chestData, g_strings->lblTagMoles, m_steamMoles);

  TNISetSoupKeyValue(io_data, g_strings->lblTagSteamChest, chestData);

  // Everything else is either from the spec, or calculated in Update().
}


//=============================================================================
// Name: Load
// Desc: Loads the internal state of the steam chest from the soup passed.
// Parm: data - The (read-only) soup to restore the steam chest state from.
//       Note that this soup may not have been created by this plugin.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void SteamChest::Load(const TNIRef<const TNISoup>& data, int dataVersion)
{
  const TNIRef<const TNISoup> chestData = TNICastSoup(TNIGetSoupValueByKey(data, g_strings->lblTagSteamChest));
  if (!chestData)
    return;

  m_steamMoles = SoupGetFloat(chestData, g_strings->lblTagMoles, m_steamMoles);


  // Sanity check the ranges on these variables.
  m_steamMoles = SanityCheckClamp(0, m_steamMoles, m_steamMoles);
}


}; // namespace SteamPhysics

