//=============================================================================
// File: SteamPhysics_SteamChest.h
// Desc: Declares SteamPhysics::SteamChest, an object representing the steam
//       chest within a steam engine.
//=============================================================================
#ifndef _SteamPhysics_SteamChest_H_
#define _SteamPhysics_SteamChest_H_
#pragma once

#include "TNIPhysicsGlobals.h"


//=============================================================================
// Desc: Steam engine components namespace. Organises the components, and lets
//       us to use more succinct names without poluting the global namespace.
namespace SteamPhysics
{

class Boiler;


//=============================================================================
// Name: SteamChest
// Desc: The steam engine steam chest. This takes on steam from the boiler
//       based on the regulator setting, where it is held at high pressure for
//       release into the pistons.
//=============================================================================
class SteamChest
{
public:

  SteamChest(Boiler* boiler, double volume, double superheatingConstant, double maxFlow);

  void Update(Boiler* boiler, float dt);

  double RemoveSteam(double moles);
  double GetSteamMoles(void) const { return m_steamMoles; }
  double GetTemperature(void) const { return m_steamTemp; }

  void SetRegulatorSetting(double value) { m_regulatorSetting = value; }

  double GetVolume(void) const { return m_specVolume; }
  double GetPressure(void);
  double GetPressurePSI(void);


  void Save(TNIRef<TNISoup>& io_data) const;
  void Load(const TNIRef<const TNISoup>& data, int dataVersion);


private:

  // These values are tracked and updated between frames, and need to be saved/loaded.
  double m_steamMoles;            // Current amount of steam

  // These values are derived each frame, and don't need to be saved/loaded.
  double m_steamTemp;             // Current steam temperature, in Kelvin

  // The regulator setting is updated every frame by the calling code.
  double m_regulatorSetting;      // The players current regulator setting

  // The below values are pulled directly from the active engine asset spec,
  // and stored here for easy reference only. They can be considered constant.
  double m_specVolume;            // Volume of the steam chest, in cubic metres (m^3)
  double m_specMaxFlowRate;       // Max rate steam can flow from the boiler (moles per second)
  double m_specSuperHeatingConstant; // Temperature difference between the boiler and steam chest

};


}; // namespace SteamPhysics


#endif // _SteamPhysics_SteamChest_H_

