//=============================================================================
// File: TNIPhysicsContext.cpp
// Desc: Defines the TNIPhysicsContext class, which hold state for vehicles
//       registered to this plugin in a particular TNI Context.
//=============================================================================
#include "TNIPhysicsContext.h"
#include "TNIContext.h"
#include "TNIStream.hpp"
#include "TNIFunctions.hpp"
#include "TNIPhysics.hpp"

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsVehicleState.h"
#include "TrainPhysicsShared.h"
#include "TrainPhysicsDCC.h"
#include "TrainPhysicsDiesel.h"
#include "TrainPhysicsSteam.h"


// Note that we rely on the using code to assert on the result of this.
#if _DEBUG
# define safe_cast_vehicle dynamic_cast
#else
# define safe_cast_vehicle static_cast
#endif

extern TNIPhysicsStrings* g_strings;


//=============================================================================
// Name: TNIPhysicsContext
// Desc: Constructs a new TNIPhysicsContext for the particular TNIContext
//       passed. See class header for more class info.
// Parm: context - The TNIContext that this is being created for.
//=============================================================================
TNIPhysicsContext::TNIPhysicsContext(TNIContext* context)
  : m_context(TNIRef<TNIContext>::Reference(context))
{
  // We don't vary our feature support for each vehicle, so we can just allocate
  // a single supported feature array and supply it for each registered vehicle.
  m_supportedFeatures = TNIAllocArrayWith1(TNIAllocLabel("<kuid:401543:1085>/core"));

  m_scriptFnPostMessage = TNIAllocLabel("PostMessage");
  m_scriptFnEcho = TNIAllocLabel("Echo");
  m_lblMsgMinor = TNIAllocLabel("minor");
  m_lblMsgSoup = TNIAllocLabel("soup");
}


//=============================================================================
// Name: ~TNIPhysicsContext
// Desc: Ensures that we clean up state for any still registered vehicles.
//       This shouldn't actually be required, and anything still registered
//       probably indicates a bug, but we still handle it just in case.
//=============================================================================
TNIPhysicsContext::~TNIPhysicsContext(void)
{
  ASSERT(m_vehicles.empty());

  // Clean up any vehicles that are still registered.
  for (auto itr = m_vehicles.end(); itr != m_vehicles.end(); ++itr)
    delete itr->second;
  m_vehicles.clear();
}



//=============================================================================
// Name: RegisterVehicle
// Desc: Registers a specific traincar with this context. This is called from
//       Trainz code when this plugin returns the highest valid result from a
//       call to "QueryProcessingPriority".
// Parm: vehicle - The vehicle being registered. We will retain a reference to
//       this internally until UnregisterVehicle() is called for this same
//       vehicle, or the context is deleted/shutdown.
//=============================================================================
void TNIPhysicsContext::RegisterVehicle(const TNIPhysicsVehicle* vehicle)
{
  // Verify that it isn't already registered somehow.
  if (m_vehicles.find(vehicle) != m_vehicles.end())
  {
    ASSERT(false);
    UnregisterVehicle(vehicle);
  }

  // Check the vehicle type and construct an appropriate state object. The
  // TNIPhysicsVehicleState adds a reference to this vehicle, but we don't
  // add another for the map key.
  if (TNIPhysicsGetVehicleEngineType(vehicle) == TNIPhysics_VehicleEngineSteam)
    m_vehicles[vehicle] = new TNIPhysicsVehicleStateSteam(vehicle);
  else
    m_vehicles[vehicle] = new TNIPhysicsVehicleState(vehicle);
}


//=============================================================================
// Name: UnregisterVehicle
// Desc: Unregisters a previously registered vehicle from this plugin context.
//       This will occur when the vehicle is deleted, or Trainz hands it over
//       to a more appropriate plugin (based on the result of a call to the
//       "QueryProcessingPriority" LibraryCall function).
// Parm: vehicle - The vehicle being unregistered. We will delete all known
//       state for this vehicle. If Trainz required anything to be saved for it
//       then SaveVehicleState() will have already been called.
//=============================================================================
void TNIPhysicsContext::UnregisterVehicle(const TNIPhysicsVehicle* vehicle)
{
  auto itr = m_vehicles.find(vehicle);
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  // Remove all record of this vehicle.
  delete itr->second;
  m_vehicles.erase(itr);
}


//=============================================================================
// Name: ProcessPhysicsCommands
// Desc: The main vehicle update function. Processes a vehicle command stream,
//       returning a command stream for the game to then process. Called in
//       response to a "ProcessPhysicsCommands" TNI library call.
// Parm: vehicle - The vehicle to process commands for.
// Parm: commandStream - The command stream to read and process.
// Retn: TNIRef<TNIStream> - A command stream for Trainz to process in return.
//=============================================================================
TNIRef<TNIStream> TNIPhysicsContext::ProcessPhysicsCommands(const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                                            const TNIRef<TNIStream>& commandStream)
{
  // The called functions will write commands for the game to this stream.
  TNIRef<TNIStream> resultsStream = TNIAllocStream(0, 512);

  // Write a vehicle update begin marker, so that Trainz knows which vehicle
  // the following commands are relevant to.
  TNIStreamWriteInt16(resultsStream, TNIP_UpdateVehicleBegin);
  TNIStreamWriteTNIObjectReference(resultsStream, vehicle);


  // Loop and process commands, until we run out.
  while (int16_t command = TNIStreamReadInt16(commandStream, 0))
  {
    switch (command)
    {
    case TNIP_UpdatePhysicsForVehicle:
      {
        // The update command takes 3 extra params, the time since the last
        // update (float), the front coupled vehicle, and the rear coupled
        // vehicle (to allow transfer of brake pressure, power, etc).
        float dt = TNIStreamReadFloat(commandStream, 0);
        TNIRef<const TNIObject> frontVehicle = TNIStreamReadTNIObjectReference(commandStream);
        TNIRef<const TNIObject> backVehicle = TNIStreamReadTNIObjectReference(commandStream);

        TNIRef<const TNIPhysicsVehicle> frontCoupledVehicle = TNICastPhysicsVehicle(frontVehicle);
        TNIRef<const TNIPhysicsVehicle> backCoupledVehicle = TNICastPhysicsVehicle(backVehicle);

        UpdatePhysicsForVehicle(vehicle, frontCoupledVehicle, backCoupledVehicle, dt, resultsStream);
      }
      break;

    case TNIP_AdjustMomentum:
      {
        // The game is notifying that a vehicle momentum has been internally
        // adjusted. Read off the new value, and notify the context.
        double newMomentum = TNIStreamReadFloat64(commandStream, 0);
        AdjustVehicleMomentum(vehicle, newMomentum);
      }
      break;

    case TNIP_FinaliseFramePhysics:
      {
        // The game is notifying that a vehicle has finished all physics
        // updates for this frame. Read final values and notify the context.
        float dt = TNIStreamReadFloat(commandStream, 0);
        double finalMomentum = TNIStreamReadFloat64(commandStream, 0);
        FinaliseVehicleFramePhysics(vehicle, finalMomentum, dt);
      }
      break;

    case TNIP_NormalisePhysicsData:
      // The game is notifying that physics state has been internally adjusted
      // and may need validation. Notify the context.
      NormalisePhysicsData(vehicle);
      break;

    case TNIP_SaveVehicleState:
      // The game is requesting that we save any plugin specific state for
      // this vehicle.
      SaveVehicleState(vehicle, resultsStream);
      break;

    case TNIP_LoadVehicleState:
      {
        // The game is requesting that we load any plugin specific state for
        // this vehicle. The passed soup may or may not have been generated
        // by this plugin.
        TNIRef<const TNIObject> obj = TNIStreamReadTNIObjectReference(commandStream);
        if (TNIRef<const TNISoup> soup = TNICastSoup(obj))
          LoadVehicleState(vehicle, soup);
        else
          ASSERT(false);
      }
      break;

    case TNIP_SteamEngine_AddCoalToFire:
      {
        // The coal add function takes a single extra parameter, being the
        // volume (not mass) of coal to transfer.
        double coalVolume = TNIStreamReadFloat64(commandStream, 0);
        SteamEngineAddCoal(vehicle, coalVolume);
      }
      break;


    case TNIP_ScriptFunctionCall:
      {
        TNIRef<const TNIObject> fnName = TNIStreamReadTNIObjectReference(commandStream);
        TNIRef<const TNIObject> fnParams = TNIStreamReadTNIObjectReference(commandStream);
        HandleScriptFunctionCall(TNICastLabel(fnName), TNICastSoup(fnParams), resultsStream);
      }
      break;

    default:
      // Unknown command. It's generally a bad idea to let this go unnoticed,
      // so log an error and assert.
      TNILogf(m_context, "TNIPhysicsCore::ProcessPhysicsCommands> Unknown command: %d", command);
      ASSERT(false);
      break;
    } // switch (command)

  } // while (int16_t command = TNIStreamReadInt16(io_commandStream, 0))


  // Finished commands, return the stream we've been writing commands to.
  return resultsStream;
}


//=============================================================================
// Name: UpdatePhysicsForVehicle
// Desc: The main physics update function. This is regularly called by Trainz
//       code to update the traincar physics for a particular time period.
//       This function takes two extra params for the front and rear coupled
//       vehicles, but keep in mind these may be using different TNI plugins.
// Parm: vehicle - The vehicle to update physics for.
// Parm: dt - The time period to update for, in seconds (ie, the time that has
//       passed since this vehicle was last updated).
// Parm: frontCoupledVehicle - The vehicle coupled to the front of the vehicle
//       being updated. This is the literal physical front, and is not altered
//       based on train controls heading.
// Parm: frontCoupledVehicle - The vehicle coupled to the rear of the vehicle
//       being updated. This is the literal physical rear, and is not altered
//       based on train controls heading.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write the results
//       of this update call. This includes the changes to vehicle momentum,
//       output of updated engine params/stats, changes to PFX, wheel
//       animation, etc.
//=============================================================================
void TNIPhysicsContext::UpdatePhysicsForVehicle(const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                                const TNIRef<const TNIPhysicsVehicle>& frontCoupledVehicle,
                                                const TNIRef<const TNIPhysicsVehicle>& backCoupledVehicle,
                                                float dt, TNIRef<TNIStream>& io_resultsStream)
{
  // Find internal data, pass that as primary param
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }


  if (!itr->second->m_bHasNotifiedSupportedFeatures)
  {
    // Write a command notifying Trainz what features we support.
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetPluginFeatureSupport);
    TNIStreamWriteTNIObjectReference(io_resultsStream, m_supportedFeatures);

    itr->second->m_bHasNotifiedSupportedFeatures = true;
  }

  double momentumReduction = DetermineMomentumReduction(itr->second, dt, io_resultsStream);

  double newMomentum = UpdateEngineForce(itr->second, dt, io_resultsStream);

  PerformMomentumReduction(itr->second, newMomentum, momentumReduction, io_resultsStream);

  UpdateControlsStateInput(itr->second, frontCoupledVehicle, backCoupledVehicle, dt, io_resultsStream);
}



//=============================================================================
// Name: DetermineMomentumReduction
// Desc: Called to determine the amount of momentum reduction for a traincar
//       at the beginning of the main update call. This includes all internal
//       engine resistances, braking resistance, slope resistance, and general
//       track resistance.
// Parm: vehicle - The vehicle to update physics for.
// Parm: dt - The time period to update for, in seconds.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
// Retn: double - The amount to reduce the passed vehicles momentum by.
//=============================================================================
double TNIPhysicsContext::DetermineMomentumReduction(TNIPhysicsVehicleState* vehicle, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  // Steam engine physics perform resistance and braking calculations within
  // the internal engine components (specifically DrivenWheel), so don't double
  // up on it here.
  if (TNIPhysicsGetVehicleEngineType(vehicle->GetVehicle()) == TNIPhysics_VehicleEngineSteam)
    return 0;

  // Both of these return a resistance value in Newtons.
  double resistanceForce = DetermineResistanceForce(vehicle->GetVehicleRef(), io_resultsStream);
  double brakingForce = DetermineBrakingForce(vehicle->GetVehicleRef(), io_resultsStream);

  double momentumReduction = (resistanceForce + brakingForce) * dt;
  return momentumReduction;
}


//=============================================================================
// Name: UpdateEngineForce
// Desc: Called to update a traincar engine and determine the force output for
//       a call to UpdatePhysicsForVehicle(). The exact behaviour will vary
//       based on the vehicle type and active control mode, but always ends in
//       returning a new desired momentum.
// Parm: vehicle - The vehicle to update physics for.
// Parm: dt - The time period to update for, in seconds.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
// Retn: double - The new momentum for the passed vehicle, as updated for
//       current engine force. The calling code may further adjust this before
//       writing it to an update command for Trainz.
//=============================================================================
double TNIPhysicsContext::UpdateEngineForce(TNIPhysicsVehicleState* vehicle, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  double wheelMomentum = 0;

  bool bIsSteamEngine = TNIPhysicsGetVehicleEngineType(vehicle->GetVehicle()) == TNIPhysics_VehicleEngineSteam;

  switch (TNIPhysicsGetVehiclePhysicsModelSetting(vehicle->GetVehicle()))
  {
  case TNIPhysics_TrainPhysicsDCC:
    wheelMomentum = UpdateEngineForceDCC(vehicle, dt, io_resultsStream);
    break;

  case TNIPhysics_TrainPhysicsRealistic:
    if (bIsSteamEngine)
      wheelMomentum = UpdateEngineForceSteam(safe_cast_vehicle<TNIPhysicsVehicleStateSteam*>(vehicle), dt, io_resultsStream);
    else
      wheelMomentum = UpdateEngineForceDieselElectric(vehicle, dt, io_resultsStream);
    break;
  }

  vehicle->m_wheelMomentumPrev = wheelMomentum;
  return wheelMomentum;
}


//=============================================================================
// Name: PerformMomentumReduction
// Desc: Called from UpdatePhysicsForVehicle() to apply the per-frame momentum
//       reduction calculated by the call to DetermineMomentumReduction().
// Parm: vehicle - The vehicle to update physics for.
// Parm: newMomentum - The new momentum for the vehicle, including any newly
//       applied engine forces. (This will not match the momentum on the passed
//       vehicle, as that won't yet include the engine output for this update.)
// Parm: momentumReduction - The amount to reduce newMomentum by, as calculated
//       by the call to DetermineMomentumReduction().
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
//=============================================================================
void TNIPhysicsContext::PerformMomentumReduction(TNIPhysicsVehicleState* vehicle,
                                                 double newMomentum, double momentumReduction,
                                                 TNIRef<TNIStream>& io_resultsStream)
{
  // Prevent the friction forces exceeding the new momentum.
  if (newMomentum > 0)
  {
    if (newMomentum < momentumReduction)
      momentumReduction = newMomentum;
  }
  else
  {
    if (newMomentum > -momentumReduction)
      momentumReduction = newMomentum;
    else
      momentumReduction = -momentumReduction;
  }


  // Write this to the result stream for processing by Trainz code.
  TNIStreamWriteInt16(io_resultsStream, TNIP_AddWheelMomentum);
  TNIStreamWriteFloat64(io_resultsStream, -momentumReduction);


  // Update our internal tracking too.
  if (TNIPhysicsGetVehiclePhysicsModelSetting(vehicle->GetVehicle()) == TNIPhysics_TrainPhysicsDCC)
    vehicle->m_wheelMomentumPrev -= momentumReduction;

  if (TNIPhysicsGetVehicleEngineType(vehicle->GetVehicle()) != TNIPhysics_VehicleEngineSteam)
  {
    // Update the wheel animation data while we're here
    double wheelVelocity = (newMomentum - momentumReduction + vehicle->m_cachedWheelslipForce) / TNIPhysicsGetVehicleMass(vehicle->GetVehicle());
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelAnimationData);
    TNIStreamWriteFloat(io_resultsStream, 0.f);
    TNIStreamWriteFloat(io_resultsStream, (float)wheelVelocity);
  }
}


//=============================================================================
// Name: AdjustVehicleMomentum
// Desc: Called from Trainz to notify of an adjustment to a vehicles momentum.
//       This is typically used to apply the results of coupler physics
//       interactions. Vehicles for which the plugin does not internally track
//       momentum can safely ignore this call.
// Parm: vehicle - The vehicle to update physics for.
// Parm: newMomentum - The new momentum to apply to the passed vehicle.
//=============================================================================
void TNIPhysicsContext::AdjustVehicleMomentum(const TNIRef<const TNIPhysicsVehicle>& vehicle, double newMomentum)
{
  // Find the internal data.
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  // We currently only use this for steam trains, where we maintain a lot of
  // internal state between each frame.
  if (TNIPhysicsGetVehicleEngineType(vehicle.c_obj()) == TNIPhysics_VehicleEngineSteam)
  {
    auto steamState = safe_cast_vehicle<TNIPhysicsVehicleStateSteam*>(itr->second);
    ASSERT(steamState);

    // Notify the steam state that the vehicle momentum has changed.
    steamState->AdjustMomentum(newMomentum);
  }
}


//=============================================================================
// Name: FinaliseVehicleFramePhysics
// Desc: Called by Trainz to notify that all updates relating to this vehicle
//       for the current frame are complete. This includes all plugin updates
//       and all coupler interactions. Plugins may use this as an appropriate
//       location to synchronise plugin state to that of the game, or apply
//       visible state changes only. At the time this call is made it is "too
//       late" to perform or apply physics calculations for a given frame.
// Parm: vehicle - The vehicle to update physics for.
// Parm: finalMomentum - The final calculated momentum for this vehicle for
//       this frame, including all plugin updates, coupler interactions, etc.
// Parm: dt - The time period used for the finalised frame, in seconds.
//=============================================================================
void TNIPhysicsContext::FinaliseVehicleFramePhysics(const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                                    double finalMomentum, float dt)
{
  // Find the internal data
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  // We currently only use this for steam trains, specifically to synchronise
  // the vehicle momentum with the final game result, and pass back the updated
  // wheel animation data.
  if (TNIPhysicsGetVehicleEngineType(vehicle.c_obj()) == TNIPhysics_VehicleEngineSteam)
  {
    auto steamState = safe_cast_vehicle<TNIPhysicsVehicleStateSteam*>(itr->second);
    ASSERT(steamState);

    steamState->FinaliseFrame(finalMomentum, dt);
  }
}


//=============================================================================
// Name: NormalisePhysicsData
// Desc: Called following an external physics state update in order to perform
//       any physics-mode-specific follow up validation/correction. Examples
//       include after physics/control state replication in multiplayer.
// Parm: vehicle - The vehicle to update physics for.
//=============================================================================
void TNIPhysicsContext::NormalisePhysicsData(const TNIRef<const TNIPhysicsVehicle>& vehicle)
{
  // Find the internal data
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  // Sync the previous momentum or it'll cause issues the next update.
  itr->second->m_wheelMomentumPrev = TNIPhysicsGetVehicleWheelMomentum(vehicle.c_obj());
}


//=============================================================================
// Name: SaveVehicleState
// Desc: Called by Trainz code to request save data for a specific vehicle.
//       If the plugin maintains any internal state for the specified vehicle
//       then this function should save that state, so that players are able
//       to save and reload with seemless physics behaviour.
//       If the plugin tracks no state for the vehicle passed, then this call
//       can be safely ignored.
// Parm: vehicle - The vehicle to save physics state for.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write the results
//       of this save call. Specifically a TNIP_ReturnSaveState command should
//       be written for any vehicles with saved state.
//=============================================================================
void TNIPhysicsContext::SaveVehicleState(const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                         TNIRef<TNIStream>& io_resultsStream)
{
  // Find the internal data
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }


  // Create a soup for the save data, and write a version number into it.
  // Note that the internal game code always writes the plugin kuid into
  // the returned data, so we do not have to save any plugin-identifying data
  // ourselves (other than the version).
  TNIRef<TNISoup> data = TNIAllocSoup();
  SoupSetInteger(data, g_strings->lblTagVersion, kSaveFormatVersion);

  // Call onto the state object to save whatever is necessary.
  itr->second->SaveVehicleState(data);

  // Write the result to the results stream.
  TNIStreamWriteInt16(io_resultsStream, TNIP_ReturnSaveState);
  TNIStreamWriteTNIObjectReference(io_resultsStream, data);
}


//=============================================================================
// Name: LoadVehicleState
// Desc: Called by Trainz code to load previously saved vehicle state.
// Parm: vehicle - The vehicle to load state for.
// Parm: data - The data to attempt to load from. Note that this save data may
//       not have been generated by this plugin (see below for more).
//=============================================================================
void TNIPhysicsContext::LoadVehicleState(const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                         const TNIRef<const TNISoup>& data)
{
  // Find the internal data.
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  bool bIsTNIPhysicsCore = true;

  // Note that the Trainz code will always include the KUID of the plugin which
  // generated this save data. To aid in 3rd party development and support the
  // core plugin does not attempt to alter it's handling of save data based on
  // this tag, but most 3rd party plugins generally should.
  TNIRef<const TNIAssetID> pluginKuid = TNICastAssetID(TNIGetSoupValueByKey(data, g_strings->lblTagPluginKUID));
  if (TNIGetAssetIDUserID(pluginKuid) != 401543 || TNIGetAssetIDContentID(pluginKuid) != 1085)
  {
    // Since we're not changing our load handling, log a warning to help with
    // diagnosing any issues that may result.
    TNILogf(m_context,  "TNIPhysicsCore> Loading save state from 3rd party plugin <kuid:%d:%d>",
                        TNIGetAssetIDUserID(pluginKuid), TNIGetAssetIDContentID(pluginKuid));
    bIsTNIPhysicsCore = false;
  }

  // Check that the version number is present, recognised and supported.
  int32_t version = SoupGetInteger(data, g_strings->lblTagVersion, 0);
  if (version <= 0 || version > kSaveFormatVersion)
  {
    // We'll attempt to load anyway, but this may cause weird results so log
    // a warning about it.
    TNILogf(m_context,  "TNIPhysicsCore> Unrecognised data version: %d", version);
    ASSERT(!bIsTNIPhysicsCore);
  }

  // Attempt to load the vehicle state.
  itr->second->LoadVehicleState(data, version);
}


//=============================================================================
// Name: SteamEngineAddCoal
// Desc: Called by Trainz code to add coal to a steam engine. For example, when
//       the player uses the manual "Shovel coal" command.
// Parm: vehicle - The vehicle the player is attempting to shovel coal into.
// Parm: coalVolume - The volume of coal to add. This is in "product units",
//       so doesn't have an exact units definition, but a certain mass is
//       assumed (see TNIPhysicsVehicleStateSteam::AddCoalToFire).
//=============================================================================
void TNIPhysicsContext::SteamEngineAddCoal(const TNIRef<const TNIPhysicsVehicle>& vehicle, double coalVolume)
{
  // Find the internal data
  auto itr = m_vehicles.find(vehicle.c_obj());
  if (itr == m_vehicles.end())
  {
    ASSERT(false);
    return;
  }

  // Make sure it's actually a steam engine.
  if (TNIPhysicsGetVehicleEngineType(vehicle.c_obj()) != TNIPhysics_VehicleEngineSteam)
  {
    ASSERT(false);
    return;
  }

  auto steamState = safe_cast_vehicle<TNIPhysicsVehicleStateSteam*>(itr->second);
  ASSERT(steamState);

  // Add the specified coal amount.
  steamState->AddCoalToFire(coalVolume);
}


//=============================================================================
// Name: HandleScriptFunctionCall
// Desc: Handler interface for Vehicle.LibraryCallPhysicsPlugin() function in
//       Trainz script. We implement a couple of example/test functions only.
//=============================================================================
void TNIPhysicsContext::HandleScriptFunctionCall(const TNIRef<const TNILabel>& fnName, const TNIRef<const TNISoup>& fnParams,
                                                 TNIRef<TNIStream>& io_resultsStream)
{
  if (fnName == m_scriptFnPostMessage)
  {
    // PostMessage test. Read the parameters and then write them to the results
    // stream under a 'PostMessage' command. The game code will then post a
    // message back to script.
    if (fnParams)
    {
      auto msgMinor = TNICastString(TNIGetSoupValueByKey(fnParams, m_lblMsgMinor));
      auto msgSoup = TNICastSoup(TNIGetSoupValueByKey(fnParams, m_lblMsgSoup));
      if (msgMinor)
      {
        TNIStreamWriteInt16(io_resultsStream, TNIP_ScriptPostMessage);
        TNIStreamWriteTNIObjectReference(io_resultsStream, msgMinor);
        TNIStreamWriteTNIObjectReference(io_resultsStream, msgSoup);
        return;
      }
    }
  }
  else if (fnName == m_scriptFnEcho)
  {
    if (fnParams)
    {
      // Echo test. Log the parameters object. This allows scripts to test
      // that the parameters are being received correctly by the plugin code.
      TNILogf(m_context, "TNIPhysicsCore::Echo>");
      TNILogObject(m_context, fnParams);
      return;
    }
  }
  else
  {
    // Unknown function.
    TNILogf(m_context, "TNIPhysicsCore> ERROR: Unknown script function: %s", TNIGetLabelDebugText(fnName));
    ASSERT(false);
    return;
  }


  // If we reach this point then we recognised the function, but the
  // parameters were invalid in some way.
  TNILogf(m_context, "TNIPhysicsCore> ERROR: Missing/invalid params for: %s", TNIGetLabelDebugText(fnName));
  if (fnParams)
    TNILogObject(m_context, fnParams);
}


