//=============================================================================
// File: TNIPhysicsContext.h
// Desc: Declares the TNIPhysicsContext class, which hold state for vehicles
//       registered to this plugin in a particular TNI Context.
//=============================================================================
#ifndef _TNIPhysicsContext_H_
#define _TNIPhysicsContext_H_
#pragma once

#include "trainznativeinterface.hpp"
#include <map>

class TNIContext;
class TNIPhysicsVehicle;
class TNIStream;
class TNIPhysicsVehicleState;
class TNISoup;


//=============================================================================
// Name: TNIPhysicsContext
// Desc: Used to hold state about registered vehicles in a particular TNI
//       context. Each context can be thought of as a 'world' in which vehicles
//       exist. Multiple contexts may exist at any given time, but do not
//       interact with each other in any way.
//       Instances of this are created and stored in a global map when the
//       plugin is loaded. See TNIPhysicsCore.cpp for further info.
//=============================================================================
class TNIPhysicsContext
{
public:

  TNIPhysicsContext(TNIContext* context);
  ~TNIPhysicsContext(void);

  void RegisterVehicle(const TNIPhysicsVehicle* vehicle);
  void UnregisterVehicle(const TNIPhysicsVehicle* vehicle);

  TNIRef<TNIStream> ProcessPhysicsCommands(const TNIRef<const TNIPhysicsVehicle>& vehicle, const TNIRef<TNIStream>& commandStream);


  // Latest/current data version for the vehicle save functions. This should
  // be incremented when making any changes to the save format, even if they
  // do not affect save/load compatibility.
  static const int16_t kSaveFormatVersion = 1;


private:

  void UpdatePhysicsForVehicle( const TNIRef<const TNIPhysicsVehicle>& vehicle,
                                const TNIRef<const TNIPhysicsVehicle>& frontCoupledVehicle,
                                const TNIRef<const TNIPhysicsVehicle>& backCoupledVehicle,
                                float dt, TNIRef<TNIStream>& io_resultsStream);

  double DetermineMomentumReduction(TNIPhysicsVehicleState* vehicle, float dt, TNIRef<TNIStream>& o_resultsStream);
  double UpdateEngineForce(TNIPhysicsVehicleState* vehicle, float dt, TNIRef<TNIStream>& o_resultsStream);
  void PerformMomentumReduction(TNIPhysicsVehicleState* vehicle, double newMomentum, double momentumReduction, TNIRef<TNIStream>& o_resultsStream);

  void AdjustVehicleMomentum(const TNIRef<const TNIPhysicsVehicle>& vehicle, double newMomentum);
  void FinaliseVehicleFramePhysics(const TNIRef<const TNIPhysicsVehicle>& vehicle, double finalMomentum, float dt);
  void NormalisePhysicsData(const TNIRef<const TNIPhysicsVehicle>& vehicle);

  void SaveVehicleState(const TNIRef<const TNIPhysicsVehicle>& vehicle, TNIRef<TNIStream>& o_resultsStream);
  void LoadVehicleState(const TNIRef<const TNIPhysicsVehicle>& vehicle, const TNIRef<const TNISoup>& data);

  void SteamEngineAddCoal(const TNIRef<const TNIPhysicsVehicle>& vehicle, double coalVolume);

  TNIRef<TNIStream> GetGameCommandsStream(void);
  void HandleScriptFunctionCall(const TNIRef<const TNILabel>& fnName, const TNIRef<const TNISoup>& fnParams,
                                TNIRef<TNIStream>& io_resultsStream);


  // The TNIContext that this TNIPhysicsContext maps to.
  TNIRef<TNIContext>    m_context;

  // An array of features supported by this plugin.
  TNIRef<TNIArray>      m_supportedFeatures;

  // Labels for possible script function calls.
  TNIRef<TNILabel>      m_scriptFnPostMessage;
  TNIRef<TNILabel>      m_scriptFnEcho;

  // Other miscellaneous labels.
  TNIRef<TNILabel>      m_lblMsgMinor;
  TNIRef<TNILabel>      m_lblMsgSoup;

  // Map of registered vehicles and their last known state. Entries are added
  // here by calls to RegisterVehicle(), and removed in UnregisterVehicle().
  std::map<const TNIPhysicsVehicle*, TNIPhysicsVehicleState*> m_vehicles;

};



#endif // _TNIPhysicsContext_H_

