//=============================================================================
// File: TNIPhysicsCore.cpp
// Desc: 
//=============================================================================
#include "TNIPhysicsCore.h"
#include "trainznativeinterface.hpp"
#include "TNIContext.h"
#include "TNIFunctions.hpp"
#include "TNILibrary.hpp"
#include "TNIStream.hpp"
#include "TNIDebug.h"
#include "TNIPhysics.hpp"

#include <mutex>
#include <map>

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsContext.h"


//=============================================================================
// Forward declarations for library init.
static TNIObject* LibraryCall(TNIContext* context, const TNILibrary* library, const TNILabel* function, const TNIObject* params);
static void ShutdownCall(TNIContext* context, TNILibrary* library);


//=============================================================================
// Global variables, including our context map and shared strings.
std::mutex                                g_contextMapLock;
std::map<TNIContext*, TNIPhysicsContext*> g_contextMap;
TNIPhysicsStrings*                        g_strings = nullptr;



//=============================================================================
// Name: TNIEntryPoint
// Desc: Plugin DLL entry point.
// Parm: context - The TNIContext which Trainz is loading the plugin into.
// Retn: TNILibrary* - A created TNILibrary instance for this plugin.
//=============================================================================
extern "C" TNI_ENTRYPOINT_DECL TNILibrary* TNIEntryPoint(TNIContext* context)
{
  TNILibraryInitData info;
  info.m_libraryVersion = 1;
  info.m_libraryCall = &LibraryCall;
  info.m_shutdownCall = &ShutdownCall;

  g_contextMapLock.lock();

  if (!g_strings)
    g_strings = new TNIPhysicsStrings();

  ASSERT(g_contextMap.find(context) == g_contextMap.end());

  TNILibrary* library = TNIAllocLibrary(context, g_strings->lblLibraryName.c_obj(), info);

  // Note: We don't reference the context entry in this map, but there's a
  // TNIRef member variable for it on TNIPhysicsContext itself.
  g_contextMap[context] = new TNIPhysicsContext(context);
  g_contextMapLock.unlock();

  return library;
}


//=============================================================================
// Name: ShutdownCall
// Desc: Cleanup function, called by Trainz on context shutdown. We use this to
//       delete our own TNIPhysicsContext from the global map.
// Parm: context - The TNIContext which Trainz is unloading the plugin from.
// Parm: TNILibrary* - The TNILibrary instance that's being deleted.
//=============================================================================
void ShutdownCall(TNIContext* context, TNILibrary* library)
{
  g_contextMapLock.lock();

  // Find and delete the associated TNIPhysicsContext in the global map.
  auto itr = g_contextMap.find(context);
  if (itr != g_contextMap.end())
  {
    delete itr->second;
    g_contextMap.erase(itr);
  }

  // If no contexts remain, perform any other global shutdown.
  if (g_contextMap.empty())
  {
    delete g_strings;
    g_strings = nullptr;
  }

  g_contextMapLock.unlock();
}


//=============================================================================
// Name: LibraryCall
// Desc: DLL library call interface. This is the main interface from Trainz to
//       this plugin, through which all our physics update calls are made.
// Parm: context - The TNIContext that Trainz is calling us from withing.
// Parm: library - A TNILibrary instance for a TNIPhysicsCore plugin that
//       Trainz want to communicate with.
// Parm: function - The function name that Trainz is calling.
// Parm: param - The function parameter(s) for the named function call. This
//       may be a single object such as a TNIPhysicsVehicle, or a TNIArray
//       containing multiple objects (varies per function).
//=============================================================================
static TNIObject* LibraryCall(TNIContext* context, const TNILibrary* library, const TNILabel* function, const TNIObject* param)
{
  TNIPhysicsContext* physicsContext = nullptr;
  g_contextMapLock.lock();
  {
    auto itr = g_contextMap.find(context);
    if (itr != g_contextMap.end())
      physicsContext = itr->second;
  }
  g_contextMapLock.unlock();

  if (!physicsContext)
  {
    ASSERT(false);
    return nullptr;
  }


  if (function == g_strings->lblInit)
  {
    // We do all initialisation in TNIEntryPoint(), ignore this.
    return nullptr;
  }

  if (function == g_strings->lblQueryProcessingPriority)
  {
    // Queries this library for the processing priority for a particular vehicle.
    // See the TNIPhysics_VehiclePriority enum in TNIPhysics.h for a list of
    // example return results. The higher the returned number, the more prefered
    // it is that this plugin process the particular vehicle. Plugins may vary
    // there return value based on any vehicle asset, engine asset, engine type
    // and/or physics mode.

    // This plugin treats all vehicles equally, so the vehicle param is unused.
    //TNIPhysicsVehicle* vehicle = TNICastPhysicsVehicle(param)

    // The core physics plugin is capable of handling any type of vehicle,
    // regardless of physics mode, etc., so just return the default priority.
    return ::TNIAllocInteger(TNIPhysics_PriorityDefault).c_obj_inc();
  }


  if (function == g_strings->lblRegisterVehicle)
  {
    // Registration uses a single param, being the relevant vehicle.
    physicsContext->RegisterVehicle(TNICastPhysicsVehicle(param));
    return nullptr;
  }


  if (function == g_strings->lblUnregisterVehicle)
  {
    // Deregistration uses a single param, being the relevant vehicle.
    physicsContext->UnregisterVehicle(TNICastPhysicsVehicle(param));
    return nullptr;
  }


  if (function == g_strings->lblProcessPhysicsCommands)
  {
    // The update function takes an array parameter, containing the vehicle to
    // update (element 0) and the commands to run (element 1).
    TNIRef<const TNIArray> paramsArray = TNICastArray(param);

    TNIRef<const TNIPhysicsVehicle> vehicle = TNICastPhysicsVehicle(TNIGetArrayElement(paramsArray, 0));
    TNIRef<TNIStream> commands = TNICastStream(TNIGetArrayElement(paramsArray, 1));

    return physicsContext->ProcessPhysicsCommands(vehicle, commands).c_obj_inc();
  }


  // Unknown function. It's generally a bad idea to let this go unnoticed, so
  // log an error and assert.
  TNILogf(context, "TNIPhysicsCore::LibraryCall> Unknown function: %s", TNIGetLabelDebugText(function));
  ASSERT(false);
  return nullptr;
}


