//=============================================================================
// File: TNIPhysicsGlobals.cpp
// Desc: Defines a few globally useful inline functions and defines, and the
//       TNIPhysicsStrings class, which holds label constants for function
//       names, soup tag names, etc.
//=============================================================================
#include "TNIPhysicsGlobals.h"
#include "TNIFunctions.hpp"



//=============================================================================
// Name: SoupSetInteger
// Desc: Sets an integer value to a soup. This is provided as a pair function
//       to SoupGetInteger(), but it doesn't do anything extra beyond what
//       TNISetSoupKeyValue() does itself.
// Parm: io_soup - (IN, OUT) The soup to save the integer value to.
// Parm: tagName - The name to save the value with.
// Parm: value - The value to save.
//=============================================================================
void SoupSetInteger(TNIRef<TNISoup>& io_soup, const TNIRef<TNILabel>& tagName, int32_t value)
{
  TNISetSoupKeyValue(io_soup, tagName, TNIAllocInteger(value));
}


//=============================================================================
// Name: SoupGetInteger
// Desc: Gets an integer value from a soup, with a fallback/failure/default
//       value to use if the tag isn't found.
// Parm: io_soup - (IN, OUT) The soup to read the integer value from.
// Parm: tagName - The name the value should be saved with.
// Parm: failureSignal - A fallback value to use if the tag is not found, or
//       is found with a different type.
// Retn: int32_t - The value that was found, or the fallback value as above.
//=============================================================================
int32_t SoupGetInteger(const TNIRef<const TNISoup>& soup, const TNIRef<TNILabel>& tagName, int32_t failureSignal)
{
  TNIRef<const TNIObject> obj = TNIGetSoupValueByKey(soup, tagName);
  if (obj)
  {
    if (TNIRef<const TNIInteger> objInt = TNICastInteger(obj))
      return (int32_t)TNIGetIntegerValue(objInt);

    // Tag was found, but wasn't an integer. This may represent a bug, or just
    // an incompatibility with the data being read (if it's from a 3rd party
    // plugin, for example). Assert for now, just to draw attention to it.
    ASSERT(false);
  }

  // Tag not found, or was found but wasn't an integer.
  return failureSignal;
}


//=============================================================================
// Name: SoupSetFloat
// Desc: Sets a float value to a soup, also performing basic float validation.
// Parm: io_soup - (IN, OUT) The soup to save the float value to.
// Parm: tagName - The name to save the value with.
// Parm: value - The value to save.
//=============================================================================
void SoupSetFloat(TNIRef<TNISoup>& io_soup, const TNIRef<TNILabel>& tagName, double value)
{
  NANCheck(value);
  TNISetSoupKeyValue(io_soup, tagName, TNIAllocFloat(value));
}


//=============================================================================
// Name: SoupGetFloat
// Desc: Gets a float value from a soup, with a fallback/failure/default value
//       to use if the tag isn't found.
// Parm: io_soup - (IN, OUT) The soup to read the float value from.
// Parm: tagName - The name the value should be saved with.
// Parm: failureSignal - A fallback value to use if the tag is not found, isn't
//       a valid number, or is found with a different type.
// Retn: int32_t - The value that was found, or the fallback value as above.
//=============================================================================
double SoupGetFloat(const TNIRef<const TNISoup>& soup, const TNIRef<TNILabel>& tagName, double failureSignal)
{
  TNIRef<const TNIObject> obj = TNIGetSoupValueByKey(soup, tagName);
  if (obj)
  {
    if (TNIRef<const TNIFloat> objFlt = TNICastFloat(obj))
    {
      double value = TNIGetFloatValue(objFlt);
      if (!std::isnan(value))
        return value;

      ASSERT_ONCE(false, s_bSoupGetFloat_NaN);
    }
    else if (TNIRef<const TNIInteger> objInt = TNICastInteger(obj))
    {
      // We also allow integers to be read as floats.
      return (double)TNIGetIntegerValue(objInt);
    }
    else
    {
      // Tag was found, but wasn't a float or an integer. This may represent a
      // bug, or just an incompatibility with the data being read (if it's from a
      // 3rd party plugin, for example). Assert for now, to draw attention to it.
      ASSERT(false);
    }
  }

  return failureSignal;
}



//=============================================================================
// Name: TNIPhysicsStrings
// Desc: Constructs the shared global strings that are used throughout this
//       plugin. A single global instance of this class is constructed when the
//       plugin is first loaded, and then deleted again when the last context
//       is destroyed.
//=============================================================================
TNIPhysicsStrings::TNIPhysicsStrings(void)
{
  // The name of this tni library
  lblLibraryName = TNIAllocLabel("TNIPhysicsCore");


  // Function names
  lblInit = TNIAllocLabel("Init");
  lblQueryProcessingPriority = TNIAllocLabel("QueryProcessingPriority");
  lblRegisterVehicle = TNIAllocLabel("RegisterVehicle");
  lblUnregisterVehicle = TNIAllocLabel("UnregisterVehicle");
  lblProcessPhysicsCommands = TNIAllocLabel("ProcessPhysicsCommands");


  // Engine param variable names
  lblFireTemperature = TNIAllocLabel("fire-temperature");
  lblCoalMass = TNIAllocLabel("coal-mass");
  lblSteamPistonCycle = TNIAllocLabel("steam-piston-cycle");


  // Steam PFX outlet variable names
  lblPFXSteamOutletStack = TNIAllocLabel("stack");
  lblPFXSteamOutletLowPressureValve = TNIAllocLabel("lowpressurevalve");
  lblPFXSteamOutletHighPressureValve = TNIAllocLabel("highpressurevalve");


  // Strings used as soup tag names
  lblTagVersion = TNIAllocLabel("version");
  lblTagPluginKUID = TNIAllocLabel("plugin-kuid");

  lblTagWheelMomentumPrev = TNIAllocLabel("wheel-momentum-prev");
  lblTagDCCAvgLoad = TNIAllocLabel("dcc-avg-load");

  lblTagFirebox = TNIAllocLabel("firebox");
  lblTagTemperature = TNIAllocLabel("temp");
  lblTagFuel = TNIAllocLabel("fuel");
  lblTagEnergy = TNIAllocLabel("energy");

  lblTagBoiler = TNIAllocLabel("boiler");
  lblTagWaterMass = TNIAllocLabel("water-m");
  lblTagWaterTemp = TNIAllocLabel("water-t");
  lblTagSteamMass = TNIAllocLabel("steam-m");
  lblTagSteamTemp = TNIAllocLabel("steam-t");
  lblTagPrevPressure = TNIAllocLabel("prev-pressure");

  lblTagSteamChest = TNIAllocLabel("steam-chest");
  lblTagMoles = TNIAllocLabel("moles");

  lblTagForce = TNIAllocLabel("force");
  lblTagPosition = TNIAllocLabel("pos");
  lblTagLeftMoles = TNIAllocLabel("l-moles");
  lblTagLeftTemp = TNIAllocLabel("l-temp");
  lblTagRightMoles = TNIAllocLabel("r-moles");
  lblTagRightTemp = TNIAllocLabel("r-temp");

  lblTagWheel = TNIAllocLabel("wheel");
  lblTagVelocity = TNIAllocLabel("velocity");
  lblTagMomentum = TNIAllocLabel("momentum");
  lblTagWheelSlip = TNIAllocLabel("slip");

  lblSafetyValveLow = TNIAllocLabel("safety-valve-low");
  lblSafetyValveHigh = TNIAllocLabel("safety-valve-high");
  lblTagTriggered = TNIAllocLabel("triggered");

  lblWaterConsumed = TNIAllocLabel("water-cons");
  lblCoalTimer = TNIAllocLabel("coal-timer");
}




