//=============================================================================
// File: TNIPhysicsGlobals.h
// Desc: Declares a few globally useful inline functions and defines, and the
//       TNIPhysicsStrings class, which holds label constants for function
//       names, soup tag names, etc.
//=============================================================================
#ifndef _TNIPhysicsGlobals_H_
#define _TNIPhysicsGlobals_H_
#pragma once

#include "TNIDebug.h"
#include "TNILabel.hpp"

#include <cmath>


//=============================================================================
// Typedefs for use in physics calculation functions. These all use the same
// storage type, and are just to help with code readability.
typedef double UnitKilograms;         // Measure of mass.
typedef double UnitMeters;            // Measure of distance.
typedef double UnitMetersPerSecond;   // Measure of velocity.
typedef double UnitNewtonsPerSecond;  // Yank. (N/s)
typedef double UnitJoules;            // Measure of energy. (Angular: J = r . Nm) (Linear: J = Nm) (J = 0.5 * kg * m/s * m/s)
typedef double UnitNewtons;           // Measure of force. (Linear: N = J / m) (N = kg . m/s/s)
typedef double UnitNewtonSeconds;     // Measure of linear momentum. (Ns = kg . m/s)
//typedef double UnitNewtonMeters;      // Measure of torque. (Linear: Nm = J) (Angular: Nm = J / r)
typedef double UnitWatts;             // Measure of energy transfer. (W = J/s) (W = Nm / s = N . m/s) (W = V.A)
typedef double UnitAmperes;           // Measure of current. (A = W / V)
typedef double UnitAmpereSeconds;     // Power usage over time.
typedef double UnitVoltage;           // Voltage. (W = V.A)


//=============================================================================
#define PI 3.141592654


//=============================================================================
#define USE_TNIPHYSICSCORE_ASSERT 1

#if USE_TNIPHYSICSCORE_ASSERT
# define ASSERT       TNI_ASSERT
# define ASSERT_ONCE  TNI_ASSERT_ONCE
#else
# define ASSERT(expression) (void)0
# define ASSERT_ONCE(expression, var) (void)0
#endif


//=============================================================================
// Name: NANCheck
// Desc: Validates that the float value passed is a valid number. If assertions
//       are enabled and these checks fail, then an breakpoint will be
//       triggered. This is called periodically to validate that we haven't
//       read in an invalid value from a save stream, or corrupted a value with
//       questionable maths.
//=============================================================================
static inline void NANCheck(double value) { ASSERT_ONCE(!std::isnan(value) && std::isfinite(value), s_bNANCheck_Failed); }


//=============================================================================
// Name: Sign
// Desc: Returns 1, -1, or 0, based on the sign of the value passed. Useful for
//       sign conversion of absolute values.
//=============================================================================
static inline double Sign(double x) { return x > 0 ? 1.0 : x < 0 ? -1.0 : 0.0; }


//=============================================================================
// Name: InterpolateOverTime
// Desc: Returns a scalar that can be used to interpolate a value over time.
// Parm: rate - The rate/speed at which to interpolate the value.
// Parm: steps - The number of steps to take right now.
// Retn: double - A scalar to multiply by the value being interpolated.
//=============================================================================
static inline float InterpolateOverTime(float rate, float steps) { return 1.f - powf(1.f - rate, steps); }
static inline double InterpolateOverTime(double rate, double steps) { return 1.0 - pow(1.0 - rate, steps); }


#pragma warning(push) // warning C4723: potential divide by 0
#pragma warning(disable: 4723)

//=============================================================================
// Name: SanityCheckClamp
// Desc: Clamps the passed value to an expected/valid range, with assertions to
//       alert the programmer if an invalid value is encountered. This is used
//       during save/load, where an out-of-range value possibly indicates a
//       programming error with the save format.
// Parm: min - The minimum valid value.
// Parm: val - The value to sanity check and clamp.
// Parm: max - The maximum valid value.
//=============================================================================
inline double SanityCheckClamp(double min, double val, double max)
{
  if (std::isnan(val) || !std::isfinite(val))
  {
    ASSERT(false);

    // Attempt to clamp to a valid range, but since val is corrupt, don't
    // assume the other input data is valid.
    if (!std::isnan(max) && std::isfinite(max))
      return max;

    if (!std::isnan(min) && std::isfinite(min))
      return min;

    return 0;
  }

  if (val < min)
  {
    // Assert, but allow some leeway for float inaccuaracy in the save format
    ASSERT(min == 0 ? false : (val / min > 0.99));
    return min;
  }

  if (val > max)
  {
    // Assert, but allow some leeway for float inaccuaracy in the save format
    ASSERT(val == 0 ? false : (max / val > 0.99));
    return max;
  }

  return val;
}

#pragma warning(pop)  // warning C4723: potential divide by 0


//=============================================================================
void SoupSetInteger(TNIRef<TNISoup>& io_soup, const TNIRef<TNILabel>& tagName, int32_t value);
int32_t SoupGetInteger(const TNIRef<const TNISoup>& soup, const TNIRef<TNILabel>& tagName, int32_t failureSignal);

void SoupSetFloat(TNIRef<TNISoup>& io_soup, const TNIRef<TNILabel>& tagName, double value);
double SoupGetFloat(const TNIRef<const TNISoup>& soup, const TNIRef<TNILabel>& tagName, double failureSignal);



//=============================================================================
// Name: TNIPhysicsStrings
// Desc: A holding place for various shared strings. A single global instance
//       of this class (g_strings) is allocated in TNIPhysicsCore.cpp.
//=============================================================================
class TNIPhysicsStrings
{
public:

  TNIPhysicsStrings(void);


  // The name of this tni library
  TNIRef<TNILabel>    lblLibraryName;


  // Function names
  TNIRef<TNILabel>    lblInit;
  TNIRef<TNILabel>    lblQueryProcessingPriority;
  TNIRef<TNILabel>    lblRegisterVehicle;
  TNIRef<TNILabel>    lblUnregisterVehicle;
  TNIRef<TNILabel>    lblProcessPhysicsCommands;


  // Engine param variable names
  TNIRef<TNILabel>    lblFireTemperature;
  TNIRef<TNILabel>    lblCoalMass;
  TNIRef<TNILabel>    lblSteamPistonCycle;


  // Steam PFX outlet variable names
  TNIRef<TNILabel>    lblPFXSteamOutletStack;
  TNIRef<TNILabel>    lblPFXSteamOutletLowPressureValve;
  TNIRef<TNILabel>    lblPFXSteamOutletHighPressureValve;
  // Note: See validation/steam-mode.txt for a list of valid variable names


  // Strings used as soup tag names
  TNIRef<TNILabel>    lblTagVersion;
  TNIRef<TNILabel>    lblTagPluginKUID;

  TNIRef<TNILabel>    lblTagWheelMomentumPrev;
  TNIRef<TNILabel>    lblTagDCCAvgLoad;

  TNIRef<TNILabel>    lblTagFirebox;
  TNIRef<TNILabel>    lblTagTemperature;
  TNIRef<TNILabel>    lblTagFuel;
  TNIRef<TNILabel>    lblTagEnergy;

  TNIRef<TNILabel>    lblTagBoiler;
  TNIRef<TNILabel>    lblTagWaterMass;
  TNIRef<TNILabel>    lblTagWaterTemp;
  TNIRef<TNILabel>    lblTagSteamMass;
  TNIRef<TNILabel>    lblTagSteamTemp;
  TNIRef<TNILabel>    lblTagPrevPressure;

  TNIRef<TNILabel>    lblTagSteamChest;
  TNIRef<TNILabel>    lblTagMoles;

  TNIRef<TNILabel>    lblTagForce;
  TNIRef<TNILabel>    lblTagPosition;
  TNIRef<TNILabel>    lblTagLeftMoles;
  TNIRef<TNILabel>    lblTagLeftTemp;
  TNIRef<TNILabel>    lblTagRightMoles;
  TNIRef<TNILabel>    lblTagRightTemp;

  TNIRef<TNILabel>    lblTagWheel;
  TNIRef<TNILabel>    lblTagVelocity;
  TNIRef<TNILabel>    lblTagMomentum;
  TNIRef<TNILabel>    lblTagWheelSlip;

  TNIRef<TNILabel>    lblSafetyValveLow;
  TNIRef<TNILabel>    lblSafetyValveHigh;
  TNIRef<TNILabel>    lblTagTriggered;

  TNIRef<TNILabel>    lblWaterConsumed;
  TNIRef<TNILabel>    lblCoalTimer;

};



#endif // _TNIPhysicsGlobals_H_

 