//=============================================================================
// File: TNIPhysicsVehicleState.cpp
// Desc: Declares the class used to track vehicle state between update calls.
//=============================================================================
#include "TNIPhysicsVehicleState.h"
#include "TNIPhysicsGlobals.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"

extern TNIPhysicsStrings* g_strings;


//=============================================================================
// Name: TNIPhysicsVehicleState
// Desc: Constructs the base vehicle state for the vehicle passed. Instances of
//       this (or inheriting classes) are created by the TNIPhysicsContext when
//       a vehicle is registered.
// Parm: vehicle - The vehicle this state is for.
//=============================================================================
TNIPhysicsVehicleState::TNIPhysicsVehicleState(const TNIPhysicsVehicle* vehicle)
  : m_vehicle(TNIRef<const TNIPhysicsVehicle>::Reference(vehicle))
{
  // Initialise the previous momentum to whatever the vehicle currently has,
  // as sudden changes to this can have pretty drastic effects. In many cases,
  // this will be immediately overridden by a call to LoadVehicleState().
  m_wheelMomentumPrev = TNIPhysicsGetVehicleWheelMomentum(vehicle);
}


//=============================================================================
// Name: ~TNIPhysicsVehicleState
// Desc: Releases the internal vehicle reference.
//=============================================================================
TNIPhysicsVehicleState::~TNIPhysicsVehicleState(void)
{
  m_vehicle = nullptr;
}


//=============================================================================
// Name: SaveVehicleState
// Desc: Saves all the state necessary to support later restoration of this
//       structure. This is used during Trainz Route/Session save, and when
//       switching a vehicle between physics plugins. This must save any and
//       all internal state which is required between frames. State save always
//       uses the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - (IN, OUT) A TNISoup to save our internal data to.
//=============================================================================
void TNIPhysicsVehicleState::SaveVehicleState(TNIRef<TNISoup>& io_data)
{
  SoupSetFloat(io_data, g_strings->lblTagWheelMomentumPrev, m_wheelMomentumPrev);
  SoupSetFloat(io_data, g_strings->lblTagDCCAvgLoad, m_dccAvgLoad);
}


//=============================================================================
// Name: LoadVehicleState
// Desc: Restores internal state using the soup passed, which is assumed to
//       have been created from an earlier call to the above save function, or
//       something which generates the same data (in another 3rd party plugin,
//       for example). Missing variables will be initialised to their defaults.
// Parm: data - (IN, OUT) The TNISoup to load our internal data from.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void TNIPhysicsVehicleState::LoadVehicleState(const TNIRef<const TNISoup>& data, int dataVersion)
{
  m_wheelMomentumPrev = SoupGetFloat(data, g_strings->lblTagWheelMomentumPrev, m_wheelMomentumPrev);
  m_dccAvgLoad = SoupGetFloat(data, g_strings->lblTagDCCAvgLoad, m_dccAvgLoad);
}

