//=============================================================================
// File: TrainPhysicsDCC.cpp
// Desc: Defines the main engine force update function for DCC trains.
//=============================================================================
#include "TrainPhysicsDCC.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsVehicleState.h"
#include "TrainPhysicsShared.h"



//=============================================================================
// Name: UpdateEngineForceDCC
// Desc: The main engine force update function for DCC trains. This is used for
//       both diesel and steam trains, and unpowered rolling stock.
// Parm: vehicleState - The vehicle to update/apply forces for.
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: io_resultsStream - (IN, OUT) A stream to write update commands to.
// Retn: double - The new momentum for the passed vehicle, as updated for
//       current engine force.
//=============================================================================
double UpdateEngineForceDCC(TNIPhysicsVehicleState* vehicleState, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  ASSERT(vehicleState);

  const TNIPhysicsVehicle* vehicle = vehicleState->GetVehicle();
  double wheelMomentum = TNIPhysicsGetVehicleWheelMomentum(vehicle);

  double mass = TNIPhysicsGetVehicleMass(vehicle);
  if (mass <= 0)
    return wheelMomentum;


  // Never use wheelslip in DCC
  if (TNIPhysicsGetVehicleWheelslip(vehicle) != 0)
  {
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelslipForce);
    TNIStreamWriteFloat64(io_resultsStream, 0);
  }

  // Add in any slope force
  double slopeForce = DetermineSlopeForce(vehicleState->GetVehicleRef(), io_resultsStream);
  wheelMomentum += slopeForce * dt;


  if (TNIPhysicsGetVehicleEngineType(vehicle) == TNIPhysics_VehicleEngineNone &&
      TNIPhysicsGetVehicleHandBrakeSetting(vehicle) <= 0)
  {
    // Free moving (un-handbraked) rolling stock, no 'engine' forces to apply
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXEnergyUseAmount);
    TNIStreamWriteFloat64(io_resultsStream, 0);

    TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelMomentum);
    TNIStreamWriteFloat64(io_resultsStream, wheelMomentum);

    return wheelMomentum;
  }


  double wheelMomentumPrev = vehicleState->m_wheelMomentumPrev;

  // Check for load caused by coupled cars adjusting momentum within the game itself.
  double load = 4.0f * ((wheelMomentum - wheelMomentumPrev) / dt) / mass;

  double oldMomentum = wheelMomentumPrev;
  NANCheck(oldMomentum);


  // The DCC throttle is logarithmic, to allow finer control at lower speed
  double engineThrottle = TNIPhysicsGetVehicleEngineThrottle(vehicle);
  double logThrottle = engineThrottle * engineThrottle;

  int8_t tractionSetting = TNIPhysicsGetVehicleTractionSetting(vehicle);

  // Determine the desired momentum from our throttle, max speed and mass
  double wantMomentum = mass * logThrottle * TNIPhysicsGetVehicleMaxSpeed(vehicle) * tractionSetting;

  // Determine the maximum acceleration/deceleration rates
  double maxAccel = TNIPhysicsGetVehicleMaxDCCAcceleration(vehicle) / mass * 4.448;
  double maxDecel = TNIPhysicsGetVehicleMaxDCCDeceleration(vehicle) / mass * 4.448;

  {
    double avgLoad = vehicleState->m_dccAvgLoad * 0.8 + load * 0.2;
    if (avgLoad < 0)
      avgLoad = (avgLoad < -1.0) ? avgLoad : -1.0;
    else
      avgLoad = (avgLoad > 1.0) ? avgLoad : 1.0;

    if (wantMomentum < 0)
    {
      if (oldMomentum > 0)
      {
        // Moving forward, dcc backward
        if (avgLoad > 0)
          maxDecel /= avgLoad;
      }
      else
      {
        // Moving backward, dcc backward
        if (avgLoad < 0)
          maxDecel /= -avgLoad;
        else
          maxAccel /= avgLoad;
      }
    }
    else
    {
      if (oldMomentum > 0)
      {
        // Moving forward, dcc forward
        if (avgLoad > 0)
          maxDecel /= avgLoad;
        else
          maxAccel /= -avgLoad;
      }
      else
      {
        // Moving backward, dcc forward
        if (avgLoad < 0)
          maxDecel /= -avgLoad;
      }
    }

    vehicleState->m_dccAvgLoad = avgLoad;
  }


  if (!TNIPhysicsGetVehicleAllowsUserControl(vehicle))
  {
    // Adjust the braking rate slightly for AI/script trains.
    if (fabs(wantMomentum) < fabs(oldMomentum) - (2.235 * mass) || (wantMomentum * oldMomentum) <= 0 || fabs(wantMomentum / mass) <= 0.2)
    {
      // Braking, fast, allow the train to slow down at 'normal' dcc rates.
      vehicleState->m_brakeSettingAIOverride = TNIPhysics_TrainBrakeApplication;
    }
    else
    {
      if (fabs(wantMomentum) < fabs(oldMomentum))
      {
        // Braking, but not quickly, reduce the deceleration rate.
        maxDecel = TNIPhysicsGetVehicleMaxDCCDeceleration(vehicle) / mass * 4.448;
      }

      vehicleState->m_brakeSettingAIOverride = TNIPhysics_TrainBrakeRelease;
    }
  }


  wheelMomentum = wantMomentum;
  NANCheck(wheelMomentum);

  // Enforce maximum acceleration/deceleration rates
  double deltaAccel = (wheelMomentum - oldMomentum);

  if (fabs(oldMomentum) < 0.1)
  {
    if (deltaAccel > maxAccel * dt * mass)
      wheelMomentum = oldMomentum + maxAccel * dt * mass;
    if (deltaAccel < -maxAccel * dt * mass)
      wheelMomentum = oldMomentum - maxAccel * dt * mass;
  }
  else if (oldMomentum > 0.0)
  {
    if (deltaAccel > maxAccel * dt * mass)
      wheelMomentum = oldMomentum + maxAccel * dt * mass;
    if (deltaAccel < -maxDecel * dt * mass)
      wheelMomentum = oldMomentum - maxDecel * dt * mass;
  }
  else
  {
    if (deltaAccel < -maxAccel * dt * mass)
      wheelMomentum = oldMomentum - maxAccel * dt * mass;
    if (deltaAccel > maxDecel * dt * mass)
      wheelMomentum = oldMomentum + maxDecel * dt * mass;
  }
  NANCheck(wheelMomentum);


  // Write the output variables.
  if (tractionSetting > 0 && deltaAccel > 0)
  {
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXEnergyUseAmount);
    TNIStreamWriteFloat64(io_resultsStream, deltaAccel);
  }
  else if (tractionSetting < 0 && deltaAccel < 0)
  {
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXEnergyUseAmount);
    TNIStreamWriteFloat64(io_resultsStream, -deltaAccel);
  }

  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelMomentum);
  TNIStreamWriteFloat64(io_resultsStream, wheelMomentum);

  return wheelMomentum;
}


