//=============================================================================
// File: TrainPhysicsDiesel.cpp
// Desc: Defines the main engine force update functions for realistice mode
//       diesel/elec trains.
//=============================================================================
#include "TrainPhysicsDiesel.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsVehicleState.h"
#include "TrainPhysicsShared.h"
#include "UnitConversion.h"
#include <algorithm>


//=============================================================================
// Name: IsRealisticPhysics
// Desc: Returns whether the vehicle passed is using realistic controls.
// Parm: vehicle - The vehicle to test.
// Retn: bool - Whether the vehicle is using the realistic physics mode.
//=============================================================================
bool IsRealisticPhysics(const TNIPhysicsVehicle* vehicle)
{
  return TNIPhysicsGetVehiclePhysicsModelSetting(vehicle) == TNIPhysics_TrainPhysicsRealistic;
}


//=============================================================================
// Name: UpdateWheelslipDieselElectric
// Desc: Performs wheelslip updates/calculations for the vehicle passed.
// Parm: vehicle - The vehicle to update the wheelslip state for.
// Retn: bool - Whether the vehicle is under wheelslip after this update.
//=============================================================================
bool UpdateWheelslipDieselElectric(const TNIPhysicsVehicle* vehicle)
{
  if (TNIPhysicsGetVehicleWheelslipTractionMultiplier(vehicle) >= 1.0)
    return false;

  bool bWantWheelslip = TNIPhysicsGetVehicleWheelslip(vehicle) > 0;
  double engForce = TNIPhysicsGetVehicleEngineForce(vehicle);
  double maxTE = TNIPhysicsGetVehicleMaximumTractiveEffort(vehicle);
  if (IsRealisticPhysics(vehicle) && TNIPhysicsGetVehicleSandingSetting(vehicle) > 0.5f)
    maxTE *= TNIPhysicsGetVehicleSandingTractionMultiplier(vehicle);

  if (bWantWheelslip)
  {
    if (fabs(engForce) < maxTE * 0.8f)
    {
      // Wheels only stop slipping when wheel motion and train motion are in
      // the same direction.
      double wheelMomentum = TNIPhysicsGetVehicleWheelMomentum(vehicle);
      if (engForce * wheelMomentum >= -1.0)
        bWantWheelslip = false;
    }
  }
  else
  {
    if (fabs(engForce) > maxTE)
      bWantWheelslip = true;
  }

  return bWantWheelslip;
}


//=============================================================================
// Name: ApplyEngineForceDieselElectric
// Desc: Called from UpdateEngineForceDieselElectric to apply the calculated
//       engine force, and update appropriate stats.
// Parm: vehicle - The vehicle to update/apply forces for.
// Parm: engineForceThisFrame - The force the engine is attempting to apply
//       this frame, in newtons.
// Parm: bWantWheelSlip - Whether the current force exceeds the maximum
//       tractive effort, and the wheels want to slip.
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: io_resultsStream - (IN, OUT) Stream to write any update commands to.
// Retn: double - The actual engine force which is being applied this frame,
//       in newtons. This will differ from the input value if the locomotive is
//       experiencing wheelslip.
//=============================================================================
double ApplyEngineForceDieselElectric(TNIPhysicsVehicleState* vehicle, double engineForceThisFrame,
                                      bool bWantWheelSlip, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  double wheelslipTractionMultiplier = TNIPhysicsGetVehicleWheelslipTractionMultiplier(vehicle->GetVehicle());

  // Update the wheelslip interpolation value.
  double wheelslip = TNIPhysicsGetVehicleWheelslip(vehicle->GetVehicle());
  if (bWantWheelSlip)
    wheelslip = 1.0;
  else
    wheelslip = std::max(wheelslip - dt, 0.0);

  // Use the wheelslip interpolation value to scale the maximum tractive effort.
  double tractionMultiplier = (1.0 - wheelslip) + (wheelslipTractionMultiplier * wheelslip);
  double maxTE = TNIPhysicsGetVehicleMaximumTractiveEffort(vehicle->GetVehicle()) * tractionMultiplier;


  // Determine the maximum force, and check if the vehicle's exceeding it.
  UnitNewtonSeconds maxForceThisFrame = maxTE * dt;
  if (fabs(engineForceThisFrame) > maxForceThisFrame)
  {
    // Wheelslip. Calculate the wheelslip force, and cap the force output.
    double forceLimitThisFrame = maxForceThisFrame * Sign(engineForceThisFrame);

    vehicle->m_cachedWheelslipForce = engineForceThisFrame - forceLimitThisFrame;
    engineForceThisFrame = forceLimitThisFrame;
  }
  else
  {
    // No wheelslip, we can apply the full engine force to forward momentum.
    vehicle->m_cachedWheelslipForce = 0;
  }


  // Write out wheelslip/engine force values for reporting in the UI etc
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelslipForce);
  TNIStreamWriteFloat64(io_resultsStream, dt > 0 ? vehicle->m_cachedWheelslipForce / dt : 0);
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetAppliedEngineForce);
  TNIStreamWriteFloat64(io_resultsStream, dt > 0 ? engineForceThisFrame / dt : 0);

  // Return the (potentially) capped engine force.
  return engineForceThisFrame;
}



//=============================================================================
// Name: UpdateEngineForceDieselElectric
// Desc: The main engine force update function for diesel/elec trains. This is
//       used exclusively for realistic mode diesel/electric trains.
// Parm: vehicleState - The vehicle to update/apply forces for.
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: io_resultsStream - (IN, OUT) The stream to write update commands to.
// Retn: double - The new momentum for the passed vehicle, as updated for
//       current engine force.
//=============================================================================
double UpdateEngineForceDieselElectric(TNIPhysicsVehicleState* vehicleState, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  ASSERT(vehicleState);
  const TNIPhysicsVehicle* vehicle = vehicleState->GetVehicle();

  double wheelMomentum = TNIPhysicsGetVehicleWheelMomentum(vehicle);
  double engineForce = 0;
  double currentDrawn = 0;
  double energyUsePFX = 0;


  // Update wheelslip.
  bool bWantWheelslip = UpdateWheelslipDieselElectric(vehicle);

  // Add in any slope force
  double slopeForce = DetermineSlopeForce(vehicleState->GetVehicleRef(), io_resultsStream) * dt;
  wheelMomentum += slopeForce;


  // Check the traction setting isn't in neutral.
  int8_t tractionSetting = TNIPhysicsGetVehicleTractionSetting(vehicle);
  if (tractionSetting != 0)
  {
    int8_t dynBrakeSetting = TNIPhysicsGetVehicleDynamicBrakeSetting(vehicle);
    if (dynBrakeSetting == TNIPhysics_DynamicBrakeTraction)
    {
      // Dynamic brake is set to traction mode.
      double maxVoltage = TNIPhysicsGetVehicleMaxVoltage(vehicle);

      UnitNewtons tractiveEffort = 1000.0f * TNIPhysicsGetVehicleThrottleGraphOutput(vehicle);
      engineForce = tractiveEffort * tractionSetting;
      UnitNewtonSeconds engineForceThisFrame = engineForce * dt;

      // Apply the calculated engine force.
      wheelMomentum += ApplyEngineForceDieselElectric(vehicleState, engineForceThisFrame, bWantWheelslip, dt, io_resultsStream);

      energyUsePFX = fabs(engineForceThisFrame);

      currentDrawn = tractiveEffort * 7.5 / maxVoltage;
    }
    else if (dynBrakeSetting == TNIPhysics_DynamicBrakeBrake)
    {
      // Dynamic brake is set to braking mode.
      double maxVoltage = TNIPhysicsGetVehicleMaxVoltage(vehicle);

      UnitNewtons tractiveEffort = 1000.0f * TNIPhysicsGetVehicleDynamicBrakeGraphOutput(vehicle);
      UnitNewtonSeconds engineForceThisFrame = tractiveEffort * dt;

      currentDrawn = std::min(tractiveEffort, fabs(wheelMomentum / dt)) * 7.5 / maxVoltage;

      if (wheelMomentum >= 0)
        engineForceThisFrame = -std::min(engineForceThisFrame, wheelMomentum);
      else
        engineForceThisFrame = std::min(engineForceThisFrame, -wheelMomentum);

      engineForce = engineForceThisFrame / dt;

      // Apply the calculated braking force.
      wheelMomentum += ApplyEngineForceDieselElectric(vehicleState, engineForceThisFrame, bWantWheelslip, dt, io_resultsStream);
    }
    else
    {
      // Dynamic brake in neutral, apply no engine force.
    }

  } // if (tractionSetting != 0)


  // Write output commands
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineForce);
  TNIStreamWriteFloat64(io_resultsStream, engineForce);
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetCurrentDrawn);
  TNIStreamWriteFloat64(io_resultsStream, currentDrawn);
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXEnergyUseAmount);
  TNIStreamWriteFloat64(io_resultsStream, energyUsePFX);

  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelMomentum);
  TNIStreamWriteFloat64(io_resultsStream, wheelMomentum);

  return wheelMomentum;
}



