//=============================================================================
// File: TrainPhysicsShared.cpp
// Desc: Defines a few shared train physics functions.
//=============================================================================
#include "TrainPhysicsShared.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsVehicleState.h"
#include "VehicleBrakeData.h"
#include "VehicleBrakePipeData.h"
#include "UnitConversion.h"
#include <algorithm>



//=============================================================================
// Name: IsStoppedDCCTrain
// Desc: Returns whether the vehicle passed is a stopped DCC train. This can be
//       used to disable certain engine physics which are not desirable in the
//       simple controls mode (like having a train roll away down a hill).
// Parm: vehicle - The vehicle to test.
// Retn: bool - Whether the vehicle passed is stopped, throttled down, and
//       using DCC/simple controls.
//=============================================================================
bool IsStoppedDCCTrain(const TNIRef<const TNIPhysicsVehicle>& vehicle)
{
  return  TNIPhysicsGetVehiclePhysicsModelSetting(vehicle.c_obj()) == TNIPhysics_TrainPhysicsDCC &&
          TNIPhysicsGetVehicleIsStopped(vehicle.c_obj()) &&
          TNIPhysicsGetVehicleEngineThrottle(vehicle.c_obj()) < 0.01f;
}


//=============================================================================
// Name: DetermineResistanceForce
// Desc: Calculates and returns the cumulative resistance forces (excluding
//       brakes) for a given vehicle. Used by everything except steam trains.
// Parm: vehicleRef - The vehicle to calculate the forces for.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
// Retn: double - The determined total resistance force, in Newtons.
//=============================================================================
double DetermineResistanceForce(const TNIRef<const TNIPhysicsVehicle>& vehicleRef, TNIRef<TNIStream>& io_resultsStream)
{
  const TNIPhysicsVehicle* vehicle = vehicleRef.c_obj();

  // The method below is used to determine the force of resistance against a
  // traincar. It uses davis' formula for determining the resistance. Davis
  // formula takes into account:
  //  * Frictional force
  //  * Track curvature force
  //  * Wind resistance
  //  * Losses varying with speed, i.e., sway, flange friction, bumps, etc.
  //
  // Apply davis formula:
  // R = (1.3 + 29.0 / w) + 0.045 * v + ((0.0005 * a) / (w * n)) * v * v;
  //
  // R = resistance (lbs / ton)
  // w = unit weight per axle
  // n = number of axles per car
  // a = effective cross sectional area of car (sq. f)
  // v = speed (miles / hr)
  //

  double velocity = TNIPhysicsGetVehicleVelocity(vehicle);
  double dv_v = UnitConversion::mps_mph(abs(velocity));

  double dv_n = TNIPhysicsGetVehicleAxleCount(vehicle);

  double vehicleMass = TNIPhysicsGetVehicleMass(vehicle);
  double weightInTons = UnitConversion::kg_ton(vehicleMass);
  double dv_w = weightInTons / dv_n;

  // 120 sq. ft. for locomotives and passenger cars, 90 sq. ft. for freight cars
  double dv_a = TNIPhysicsGetVehicleSurfaceArea(vehicle);

  // 0.03 for locomotives, 0.045 freight cars
  double movingFrictionCoefficient = TNIPhysicsGetVehicleMovingFrictionCoefficient(vehicle);

  // 0.0017 for streamlined locos, 0.0025 for other locos, 0.0005 for trailing freight cars, 0.00034 for trailing passenger cars
  double airDragCoefficient = TNIPhysicsGetVehicleAirDragCoefficient(vehicle);

  double d_curve = TNIPhysicsGetVehicleTrackCurveAngle(vehicle);

  // Determine ground resistance as per davis-formula
  double resistanceForce = 1.3 + 29.0 / dv_w + movingFrictionCoefficient * dv_v + ((airDragCoefficient * dv_a) / (dv_w * dv_n)) * dv_v * dv_v;

  // Add curve resistance
  resistanceForce += (dv_v > UnitConversion::mph_mps(2.0) ? 0.8 : 1.0) * d_curve;

  resistanceForce *= weightInTons;

  // Convert to correct (SI) units (i.e. Newtons)
  resistanceForce = UnitConversion::lbf_kn(resistanceForce) * 1000;

  // Save this out for reporting in the UI etc.
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetResistanceForce);
  TNIStreamWriteFloat64(io_resultsStream, resistanceForce);

  // Final result.
  return resistanceForce;
}


//=============================================================================
// Name: DetermineSlopeForce
// Desc: Calculates and returns the force of gravity pulling the traincar down
//       sloped track. This may either slow or speed up the train.
// Parm: vehicleRef - The vehicle to calculate the slope force for.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
// Retn: double - The determined slope force, in Newtons.
//=============================================================================
double DetermineSlopeForce(const TNIRef<const TNIPhysicsVehicle>& vehicleRef, TNIRef<TNIStream>& io_resultsStream)
{
  const TNIPhysicsVehicle* vehicle = vehicleRef.c_obj();

  // Determine slope force
  double gradient = TNIPhysicsGetVehicleTrackGradient(vehicle);
  double vehicleMass = TNIPhysicsGetVehicleMass(vehicle);

  double slopeForce = gradient * vehicleMass * -9.8;

  // Make it easier to stop DCC trains on steep slopes by gradually eliminating
  // the slope force. This also prevents stationary trains from rolling away.
  if (TNIPhysicsGetVehiclePhysicsModelSetting(vehicle) == TNIPhysics_TrainPhysicsDCC &&
      TNIPhysicsGetTrainLocoCount(vehicle) > 0 && TNIPhysicsGetVehicleEngineThrottle(vehicle) < 0.01f)
  {
    double speed = abs(TNIPhysicsGetVehicleVelocity(vehicle));
    if (speed < 1.0)
      slopeForce *= speed;
  }

  // TODO: Add a stat for this?
  return slopeForce;
}


//=============================================================================
// Name: DetermineBrakingForce
// Desc: Calculates and returns the braking force for a given vehicle. Used
//       only by non-steam realistic mode trains.
// Parm: vehicleRef - The vehicle to calculate the slope force for.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
// Retn: double - The determined braking force, in Newtons.
//=============================================================================
double DetermineBrakingForce(const TNIRef<const TNIPhysicsVehicle>& vehicleRef, TNIRef<TNIStream>& io_resultsStream)
{
  const TNIPhysicsVehicle* vehicle = vehicleRef.c_obj();

  // This is unused in DCC physics mode
  if (TNIPhysicsGetVehiclePhysicsModelSetting(vehicle) == TNIPhysics_TrainPhysicsDCC)
    return 0;

  const double abP = TNIPhysicsGetVehicleAutoBrakeCylinderPressure(vehicle);
  const double ibP = TNIPhysicsGetVehicleIndependentBrakeCylinderPressure(vehicle);
  const double sbP = TNIPhysicsGetVehicleScriptBrakePressure(vehicle);

  double brakePressure = UnitConversion::gm3_PSI(std::max(sbP, std::max(abP, ibP)));
  if (brakePressure < 0)
    brakePressure = 0;

  // 20420 lbs at 50psi - includes braking ratio & rigging efficiency
  double brakeRatio = TNIPhysicsGetEngineBrakeRatio(vehicle) / 50.0;

  // Coefficient of friction between brake shoe and wheel
  const double shoeFriction = 0.3;

  double brakingForce = brakeRatio * brakePressure * shoeFriction;

  // Convert to correct (SI) units (i.e. Newtons)
  brakingForce = UnitConversion::lbf_kn(brakingForce) * 1000;

  // Save this out for reporting in the UI etc.
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetBrakingForce);
  TNIStreamWriteFloat64(io_resultsStream, brakingForce);

  // Return final result.
  return brakingForce;
}



//=============================================================================
// Name: UpdateControlsStateInput
// Desc: Updates control input, brake pressures, and brake pipe connections.
//       This is used by everything, regardless of control mode or engine type.
// Parm: vehicleState - The vehicle to update the controls for.
// Parm: frontCoupledVehicleRef - A reference to the vehicle coupled to the
//       front of the vehicle being updated, if any. (Used to update connected
//       systems, like brake pipes.)
// Parm: frontCoupledVehicleRef - A reference to the vehicle coupled to the
//       rear of the vehicle being updated, if any. (Used to update connected
//       systems, like brake pipes.)
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: io_resultsStream - (IN, OUT) A stream onto which to write commands
//       for Trainz to process in response to this call.
//=============================================================================
void UpdateControlsStateInput(TNIPhysicsVehicleState* vehicleState,
                              const TNIRef<const TNIPhysicsVehicle>& frontCoupledVehicleRef,
                              const TNIRef<const TNIPhysicsVehicle>& backCoupledVehicleRef,
                              float dt, TNIRef<TNIStream>& io_resultsStream)
{
  FixedPneumatic atmosphere(kAtmosphericPressureGM3);

  const TNIPhysicsVehicle* vehicle = vehicleState->GetVehicle();
  int8_t physicsMode = TNIPhysicsGetVehiclePhysicsModelSetting(vehicle);

  float brakeReleaseSound = TNIPhysicsGetVehicleBrakeReleaseSoundFade(vehicle);
  brakeReleaseSound *= 0.9f;

  float brakePipeSpecPressure = TNIPhysicsGetVehiclePressure_BrakePipe(vehicle);

  // Construct brake data from this vehicle.
  VehicleBrakeData brakes(vehicle);
  VehicleBrakePipeData brakePipes(brakePipeSpecPressure, vehicle);

  const double mainReservoirPressureStart = brakes.mainReservoir.GetPressure();


  if (brakes.equaliser.Exists())
  {
    // Grab the current brake setting
    float brakeSetting;
    if (vehicleState->m_brakeSettingAIOverride >= 0)
    {
      // We're overriding the brake setting for an AI controlled train.
      // Do this for a single update only, then clear the flag to prevent the
      // brakes ever getting "stuck on".
      brakeSetting = (float)vehicleState->m_brakeSettingAIOverride;
      vehicleState->m_brakeSettingAIOverride = -1;
    }
    else
    {
      // Use the brake setting as specified by Trainz.
      brakeSetting = TNIPhysicsGetVehicleTrainBrakeSetting(vehicle);
    }

    switch ((int)brakeSetting)
    {
    case TNIPhysics_TrainBrakeHandleOff:
      // Vent the equaliser to the atmosphere.
      brakes.equaliser.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_EqualiserVentHandleOff(vehicle), dt);
      break;

    case TNIPhysics_TrainBrakeRelease:
      // The equalising reservoir is charged to the same pressure as the brake
      // pipe when the brakes are released.
      if (brakes.equaliser.GetPressure() < brakePipeSpecPressure)
      {
        brakes.equaliser.Exchange(brakes.mainReservoir, TNIPhysicsGetVehicleFlowsize_EqualiserMainReservoir(vehicle), dt);
        brakeReleaseSound += 0.1f;
      }
      break;

    case TNIPhysics_TrainBrakeInitial:
      {
        // When the train brake is applied, pressure is reduced to the desired
        // level in the equalising reservoir.
        float pressureInitial = TNIPhysicsGetVehiclePressure_BrakeInitial(vehicle);
        float pressureRange = TNIPhysicsGetVehiclePressure_BrakeFull(vehicle) - pressureInitial;
        float pressure = pressureInitial + pressureRange * (brakeSetting - TNIPhysics_TrainBrakeInitial);
        if (brakes.equaliser.GetPressure() > pressure)
        {
          brakes.equaliser.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_EqualiserVent(vehicle), dt);
          brakeReleaseSound += 0.1f;
        }
      }
      break;

    case TNIPhysics_TrainBrakeApplication:
      // When the train brake is applied, pressure is reduced to the desired
      // level in the equalising reservoir.
      if (brakes.equaliser.GetPressure() > TNIPhysicsGetVehiclePressure_BrakeFull(vehicle))
      {
        brakes.equaliser.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_EqualiserVent(vehicle), dt);
        brakeReleaseSound += 0.1f;
      }
      break;

    case TNIPhysics_TrainBrakeEmergency:
      // In emergency, rapidly vent all equaliser pressure to the atmosphere.
      brakes.equaliser.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_EqualiserVentEmergency(vehicle), dt);
      break;

    case TNIPhysics_TrainBrakeLap:
      // In the lap position, the equalising reservoir pressure is retained.
      // The distributing valve then reduces the brake pipe pressure until it
      // matches the equaliser.
      break;
    }
  }

  if (brakePipes.no3Pipe.Exists())
  {
    // Independent brake. Determine the desired pressure for the current lever position.
    float locoBrakeSetting = TNIPhysicsGetVehicleLocoBrakeSetting(vehicle);
    double wantPressure;
    if ((int)locoBrakeSetting == 0)
      wantPressure = kAtmosphericPressureGM3;
    else
      wantPressure = TNIPhysicsGetVehiclePressure_IndBrakeFull(vehicle) * locoBrakeSetting / 32.f;

    // Update the pressure in the no3/control pipe as desired. The independent
    // brake cylinder will be updated to match (further below).
    if (wantPressure > brakePipes.no3Pipe.GetPressure() * 1.01)
    {
      // Feed air directly from the main reservoir to the control pipe. This
      // runs via the independent brake controller and a double-check valve,
      // preventing back-flow.
      if (brakes.mainReservoir.GetPressure() > brakePipes.no3Pipe.GetPressure())
        brakePipes.no3Pipe.EqualisedExchange(brakes.mainReservoir, TNIPhysicsGetVehicleFlowsize_No3PipeMainReservoir(vehicle), dt, wantPressure);
    }
    else if (wantPressure < brakePipes.no3Pipe.GetPressure())
    {
      // Vent from the pipe to atmosphere.
      brakePipes.no3Pipe.EqualisedExchange(atmosphere, TNIPhysicsGetVehicleFlowsize_No3PipeVent(vehicle), dt, wantPressure);
    }
  }


#if ENABLE_VACUUM_BRAKES
  if (brakePipes.vacuumBrakePipe.Exists())
  {
    FixedPneumatic highSpeedExhauster(TNIPhysicsGetVehiclePressure_HighSpeedExhauster(vehicle));

    float vacBrakeSetting = TNIPhysicsGetVehicleVacuumBrakeSetting(vehicle);
    switch ((int)vacBrakeSetting)
    {
    case vacuumBrakeApplication:
      highSpeedExhauster.Exchange(brakePipes.vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_HighSpeedExhausterVacuumBrakePipe(vehicle), dt);
      break;
    case vacuumBrakeNeutral:
      //lowSpeedExhauster.Exchange(brakePipes.vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_LowSpeedExhausterVacuumBrakePipe(vehicle), dt);
      break;
    case vacuumBrakeRelease:
      brakePipes.vacuumBrakePipe.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_VacuumBrakePipeReleaseVent(vehicle), dt);
      break;
    }

    if (brakePipes.vacuumBrakePipe.GetPressure() < Conversion::hg_kpa(21))
      brakePipes.vacuumBrakePipe.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_VacuumBrakePipeVent(vehicle), dt);
  }
#endif // ENABLE_VACUUM_BRAKES


  if (brakes.equaliser.Exists())
  {
    // Distributing valve- matches brake pipe pressure to the equaliser. As
    // with the independent brake above (no3 pipe), we either feed air in from
    // the main reservoir or release it into the atmosphere.
    double equaliserPressure = brakes.equaliser.GetPressure();

    if (equaliserPressure > brakePipes.trainBrakePipe.GetPressure() * 1.01)
      brakePipes.trainBrakePipe.EqualisedExchange(brakes.mainReservoir, TNIPhysicsGetVehicleFlowsize_TrainBrakePipeReservoir(vehicle), dt, equaliserPressure);
    else if (equaliserPressure < brakePipes.trainBrakePipe.GetPressure())
      brakePipes.trainBrakePipe.EqualisedExchange(atmosphere, TNIPhysicsGetVehicleFlowsize_TrainBrakePipeVent(vehicle), dt, equaliserPressure);
  }

  if (brakePipes.epReservoirPipe.Exists())
    brakes.mainReservoir.Exchange(brakePipes.epReservoirPipe, TNIPhysicsGetVehicleFlowsize_MainReservoirEP(vehicle), dt);


  // Update engine throttle.
  {
    // Interpolate this engine's throttle towards the the train's overall throttle setting.
    double adjustSpeed = (double)TNIPhysicsGetVehicleThrottleAdjustmentRate(vehicle);

    // Adjustment speed must be at least 0.1 for DCC behavior.
    if (physicsMode == TNIPhysics_TrainPhysicsDCC)
      adjustSpeed = std::max(0.1, adjustSpeed);

    float throttleSetting = TNIPhysicsGetVehicleThrottleSetting(vehicle);
    double engineThrottle = TNIPhysicsGetVehicleEngineThrottle(vehicle);

    engineThrottle += (throttleSetting - engineThrottle) * std::min(1.0, InterpolateOverTime(adjustSpeed, dt / 0.03));

    TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineThrottle);
    TNIStreamWriteFloat64(io_resultsStream, engineThrottle);
  }


  {
    int8_t epBrakeSetting = TNIPhysicsGetVehicleEpBrakeSetting(vehicle);

    // Relay Valve / Triple Valve
    if (brakePipes.no4Pipe.GetPressure() > kAtmosphericPressureGM3 * 2.0)
    {
      // Bailing off
      if (brakes.autoBrakeCylinder.GetPressure() > brakePipes.no3Pipe.GetPressure() * 1.01)
      {
        // Vent from the autoBrakeCylinder until it matches no3Pipe
        brakes.autoBrakeCylinder.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_AuxReservoirVent(vehicle), dt);
      }
      else
      {
        // Control from no3Pipe
        brakes.autoBrakeCylinder.Exchange(brakePipes.no3Pipe, TNIPhysicsGetVehicleFlowsize_AuxReservoirToNo3Pipe(vehicle), dt);
      }
    }
    else
    {
      if (brakePipes.trainBrakePipe.GetPressure() > brakes.auxReservoir.GetPressure() * 1.01)
      {
        // As brake pipe pressure is increased (brakes released), use the air
        // to recharge the auxiliary reservoir.
        brakes.auxReservoir.Exchange(brakePipes.trainBrakePipe, TNIPhysicsGetVehicleFlowsize_AuxReservoirToTrainBrakePipe(vehicle), dt);

        // At the same time, reduce brake cylinder pressure by allowing air to
        // vent to the atmosphere. The distributor will only allow air out of
        // the brake cylinder while the auxiliary reservoir is being recharged.
        {
          bool epCloseABCVent = epBrakeSetting == TNIPhysics_EPBrakeApplied || epBrakeSetting == TNIPhysics_EPBrakeNeutral;
          if (!epCloseABCVent)
            brakes.autoBrakeCylinder.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_AutoBrakeCylinderVent(vehicle), dt);
        }
      }
      else if (brakePipes.trainBrakePipe.GetPressure() * 1.01 < brakes.auxReservoir.GetPressure())
      {
        // When pipe pressure is reduced (brake application) the distributor
        // valve senses this drop in pressure and allows air to enter the brake
        // cylinder from the auxiliary reservoir. The higher the pressure drop,
        // the more air enters the cylinder, and the higher the braking force.
        brakes.auxReservoir.EqualisedExchange(brakes.autoBrakeCylinder, TNIPhysicsGetVehicleFlowsize_AuxReservoirToAutoBrakeCylinder(vehicle), dt, brakePipes.trainBrakePipe.GetPressure());

        // Deal with the DCC overloading the brake cylinders with EP pressure.
        if (epBrakeSetting != TNIPhysics_EPBrakeDisabled)
          brakes.autoBrakeCylinder.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_AutoBrakeCylinderVent(vehicle), dt);
      }
    }

    // Independent brake- match the air pressure in the cylinder to that of the
    // no3/control pipe. This functions by feeding air in from the pipe, or by
    // releasing it to the atmosphere.
    if (brakePipes.no3Pipe.GetPressure() > brakes.independentBrakeCylinder.GetPressure() * 1.01)
      brakes.independentBrakeCylinder.EqualisedExchange(brakePipes.no3Pipe, TNIPhysicsGetVehicleFlowsize_No3PipeToAutoBrakeCylinder(vehicle), dt, brakePipes.no3Pipe.GetPressure());
    else if (brakePipes.no3Pipe.GetPressure() * 1.01 < brakes.independentBrakeCylinder.GetPressure())
      brakes.independentBrakeCylinder.EqualisedExchange(atmosphere, TNIPhysicsGetVehicleFlowsize_AutoBrakeCylinderVent(vehicle), dt, brakePipes.no3Pipe.GetPressure());

    if (epBrakeSetting == TNIPhysics_EPBrakeApplied)
      brakePipes.epReservoirPipe.Exchange(brakes.autoBrakeCylinder, TNIPhysicsGetVehicleFlowsize_EpReservoirPipeToAutoBrakeCylinder(vehicle), dt);


    // TODO: Hacked this in to deal with overpressurisation from EP application,
    // but what is the real solution? Perhaps EP pressure goes into the aux
    // reservoir instead of direct to the brakes? Or replaces the flow from
    // reservoir->cylinder but is still pressure limited by the pipe?
    if (epBrakeSetting == TNIPhysics_EPBrakeReleased)
      brakes.autoBrakeCylinder.Exchange(atmosphere, TNIPhysicsGetVehicleFlowsize_AutoBrakeCylinderVent(vehicle), dt * 5.0f);
    // The 5x was added since DCC is the only thing which uses EP brakes currently, and we want faster response time


    #if ENABLE_VACUUM_BRAKES
      // TODO: Perform vacuum operations here
      if (brakePipes.vacuumBrakePipe.GetPressure() < brakes.vacuumReservoir.GetPressure())
        brakes.vacuumReservoir.Exchange(vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_VacuumBrakeReservoirToVacuumBrakePipe(vehicle), dt);

      brakes.vacuumBrakeCylinder.Exchange(brakePipes.vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_VacuumBrakeCylinderToVacuumBrakePipe(vehicle), dt);
    #endif
  }

  if (dt > 0)
  {
    // Update brake pressure flow stat before we pump more air into the system.
    const double mainReservoirPressureEnd = brakes.mainReservoir.GetPressure();
    double frameFlowRate = (mainReservoirPressureStart - mainReservoirPressureEnd) / dt;

    TNIStreamWriteInt16(io_resultsStream, TNIP_SetBrakePressureFlowRate);
    TNIStreamWriteFloat64(io_resultsStream, frameFlowRate);
  }


  FixedPneumatic compressor(TNIPhysicsGetVehiclePressure_Compressor(vehicle));

  if (brakes.mainReservoir.Exists())
  {
    // Check if we want to run the compressor to repressurise the brakes.
    bool compressorActive = TNIPhysicsGetVehicleCompressorActive(vehicle);
    if (compressorActive)
    {
      if (brakes.mainReservoir.GetPressure() > UnitConversion::PSI_gm3(140))
        compressorActive = false;

      float compressorFlow = TNIPhysicsGetVehicleFlowsize_Compressor(vehicle) * TNIPhysicsGetVehicleCompressorEfficiency(vehicle);
      compressor.Exchange(brakes.mainReservoir, compressorFlow, dt);
    }
    else
    {
      // TODO: Compressor should only run while train is turned on
      if (brakes.mainReservoir.GetPressure() < UnitConversion::PSI_gm3(130))
        compressorActive = true;
    }

    TNIStreamWriteInt16(io_resultsStream, TNIP_SetCompressorActive);
    TNIStreamWriteInt8(io_resultsStream, compressorActive ? 1 : 0);
  }



  // Update the no4 pipe for bail-off control.
  float timeSinceBail = TNIPhysicsGetVehicleBailTimeSetting(vehicle);
  if (timeSinceBail < 1.0)
    compressor.Exchange(brakePipes.no4Pipe, 1.0f, dt);

  atmosphere.Exchange(brakePipes.no4Pipe, 1.0f, dt);


  vehicleState->UpdateEnginePhysics(dt, &brakes, io_resultsStream);


  // Update coupled brake pipe connections (if there's anything connected),
  // and both cars have the necessary pipes.
  if (brakePipeSpecPressure > 0)
  {
    const TNIPhysicsVehicle* frontCoupledVehicle = frontCoupledVehicleRef.c_obj();
    if (frontCoupledVehicle && TNIPhysicsGetVehiclePressure_BrakePipe(frontCoupledVehicle) <= 0)
      frontCoupledVehicle = nullptr;
    const TNIPhysicsVehicle* backCoupledVehicle = backCoupledVehicleRef.c_obj();
    if (backCoupledVehicle && TNIPhysicsGetVehiclePressure_BrakePipe(backCoupledVehicle) <= 0)
      backCoupledVehicle = nullptr;

    if (frontCoupledVehicle || backCoupledVehicle)
    {
      VehicleBrakePipeData* frontCar = nullptr;
      if (frontCoupledVehicle)
        frontCar = new VehicleBrakePipeData(brakePipeSpecPressure, frontCoupledVehicle);
      VehicleBrakePipeData* backCar = nullptr;
      if (backCoupledVehicle)
        backCar = new VehicleBrakePipeData(brakePipeSpecPressure, backCoupledVehicle);


      if (brakePipes.trainBrakePipe.Exists())
      {
        // leaky train-brake-pipe coupling(s)?
        float scriptPipeEfficiency = TNIPhysicsGetVehicleBrakePipeEfficiency(vehicle);
        if (frontCoupledVehicle && scriptPipeEfficiency < 1.0f)
        {
          double leakRate = TNIPhysicsGetVehicleFlowsize_EqualiserVentEmergency(vehicle) * (1.0 - scriptPipeEfficiency);
          if (frontCoupledVehicle)
            brakePipes.trainBrakePipe.Exchange(atmosphere, leakRate, dt);
          if (backCoupledVehicle)
            brakePipes.trainBrakePipe.Exchange(atmosphere, leakRate, dt);
        }

        if (frontCar && frontCar->trainBrakePipe.Exists())
          brakePipes.trainBrakePipe.Exchange(frontCar->trainBrakePipe, TNIPhysicsGetVehicleFlowsize_TrainBrakePipe(vehicle), dt, frontCar->pressureRatio);
        if (backCar && backCar->trainBrakePipe.Exists())
          brakePipes.trainBrakePipe.Exchange(backCar->trainBrakePipe, TNIPhysicsGetVehicleFlowsize_TrainBrakePipe(vehicle), dt, backCar->pressureRatio);
      }

      if (brakePipes.no3Pipe.Exists())
      {
        if (frontCar && frontCar->no3Pipe.Exists())
          brakePipes.no3Pipe.Exchange(frontCar->no3Pipe, TNIPhysicsGetVehicleFlowsize_No3Pipe(vehicle), dt, frontCar->pressureRatio);
        if (backCar && backCar->no3Pipe.Exists())
          brakePipes.no3Pipe.Exchange(backCar->no3Pipe, TNIPhysicsGetVehicleFlowsize_No3Pipe(vehicle), dt, backCar->pressureRatio);
      }

      if (brakePipes.no4Pipe.Exists())
      {
        if (frontCar && frontCar->no4Pipe.Exists())
          brakePipes.no4Pipe.Exchange(frontCar->no4Pipe, TNIPhysicsGetVehicleFlowsize_No4Pipe(vehicle), dt, frontCar->pressureRatio);
        if (backCar && backCar->no4Pipe.Exists())
          brakePipes.no4Pipe.Exchange(backCar->no4Pipe, TNIPhysicsGetVehicleFlowsize_No4Pipe(vehicle), dt, backCar->pressureRatio);
      }

      if (brakePipes.epReservoirPipe.Exists())
      {
        if (frontCar && frontCar->epReservoirPipe.Exists())
          brakePipes.epReservoirPipe.Exchange(frontCar->epReservoirPipe, TNIPhysicsGetVehicleFlowsize_EpReservoirPipe(vehicle), dt, frontCar->pressureRatio);
        if (backCar && backCar->epReservoirPipe.Exists())
          brakePipes.epReservoirPipe.Exchange(backCar->epReservoirPipe, TNIPhysicsGetVehicleFlowsize_EpReservoirPipe(vehicle), dt, backCar->pressureRatio);
      }

      #if ENABLE_VACUUM_BRAKES
        if (brakePipes.vacuumBrakePipe.Exists())
        {
          if (frontCar && frontCar->vacuumBrakePipe.Exists())
            brakePipes.vacuumBrakePipe.Exchange(frontCar->vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_VacuumBrakePipe(vehicle), dt, pressureRatioFront);
          if (backCar && backCar->vacuumBrakePipe.Exists())
            brakePipes.vacuumBrakePipe.Exchange(backCar->vacuumBrakePipe, TNIPhysicsGetVehicleFlowsize_VacuumBrakePipe(vehicle), dt, pressureRatioBack);
        }
      #endif


      if (frontCar)
      {
        TNIStreamWriteInt16(io_resultsStream, TNIP_UpdateVehicleBegin);
        TNIStreamWriteTNIObjectReference(io_resultsStream, frontCar->thisVeh);
        frontCar->WriteUpdateCommandsToStream(io_resultsStream);
        delete frontCar;
      }
      if (backCar)
      {
        TNIStreamWriteInt16(io_resultsStream, TNIP_UpdateVehicleBegin);
        TNIStreamWriteTNIObjectReference(io_resultsStream, backCar->thisVeh);
        backCar->WriteUpdateCommandsToStream(io_resultsStream);
        delete backCar;
      }

      // Switch back to updating the main vehicle
      TNIStreamWriteInt16(io_resultsStream, TNIP_UpdateVehicleBegin);
      TNIStreamWriteTNIObjectReference(io_resultsStream, vehicle);
    }
  }


  // Write brake system outputs
  brakes.WriteUpdateCommandsToStream(io_resultsStream);
  brakePipes.WriteUpdateCommandsToStream(io_resultsStream);

  TNIStreamWriteInt16(io_resultsStream, TNIP_SetBrakeReleaseSoundFade);
  TNIStreamWriteFloat(io_resultsStream, brakeReleaseSound);

}


