//=============================================================================
// File: TrainPhysicsSteam.cpp
// Desc: Defines the main engine force update function for realistic mode
//       steam trains, and TNIPhysicsVehicleStateSteam, which is used to track
//       between-frame state for steam engines (in both dcc and realistic).
//=============================================================================
#include "TrainPhysicsSteam.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"

#include "TNIPhysicsGlobals.h"
#include "TNIPhysicsVehicleState.h"
#include "VehicleBrakeData.h"
#include "SteamPhysicsComponents.h"
#include "UnitConversion.h"
#include <algorithm>


extern TNIPhysicsStrings* g_strings;

using namespace SteamPhysics;


//=============================================================================
// Name: UpdateEngineForceSteam
// Desc: The main engine force update function for realistic mode steam trains.
//       For steam trains this function doesn't actually do much, just grabbing
//       forward momentum that was calculated last frame and returning it. The
//       engine physics updates are actually performed in UpdateEnginePhysics()
//       below, which is called from UpdateControlsStateInput().
// Parm: vehicleState - The vehicle to update/apply forces for.
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: io_resultsStream - (IN, OUT) The stream to write update commands to.
// Retn: double - The new momentum for the passed vehicle, as updated for
//       current engine force.
//=============================================================================
double UpdateEngineForceSteam(TNIPhysicsVehicleStateSteam* vehicleState, float dt, TNIRef<TNIStream>& io_resultsStream)
{
  ASSERT(vehicleState);
  const TNIPhysicsVehicle* vehicle = vehicleState->GetVehicle();

  double wheelMomentum = vehicleState->GetForwardMomentum();
  NANCheck(wheelMomentum);

  vehicleState->UpdateMass(TNIPhysicsGetVehicleMass(vehicle));

  return wheelMomentum;
}


//=============================================================================
// Name: TNIPhysicsVehicleStateSteam
// Desc: Initialises the various steam engine components from the engine spec
//       of the vehicle passed. We track a *lot* of data between frames for
//       steam engines, and this class is responsible for allocating, managing,
//       saving and reloading it.
//       See TNIPhysicsVehicleState for more (general) info.
// Parm: vehicle - The vehicle this state is for.
//=============================================================================
TNIPhysicsVehicleStateSteam::TNIPhysicsVehicleStateSteam(const TNIPhysicsVehicle* vehicle)
  : TNIPhysicsVehicleState(vehicle)
{
  // Firebox
  m_firebox = new FireBox(TNIPhysicsGetSteamEngineBurnRateIdle(vehicle),
                          TNIPhysicsGetSteamEngineMinFireTemperature(vehicle),
                          TNIPhysicsGetSteamEngineBurnRate(vehicle),
                          TNIPhysicsGetSteamEngineMaxFireTemperature(vehicle),
                          TNIPhysicsGetSteamEngineTestSpeed(vehicle),
                          TNIPhysicsGetSteamEngineTestCutoff(vehicle),
                          TNIPhysicsGetSteamEngineIdealCoalMass(vehicle),
                          TNIPhysicsGetSteamEngineSafetyValveLowPressure(vehicle));

  m_firebox->SetupBlowerParameters(TNIPhysicsGetSteamEngineBlowerEffect(vehicle),
                                   TNIPhysicsGetSteamEngineBlowerMaxFlow(vehicle));

  m_firebox->SetupOtherParameters(TNIPhysicsGetSteamEngineFireboxEfficiency(vehicle),
                                  TNIPhysicsGetSteamEngineFireboxHeatingSurfaceArea(vehicle),
                                  TNIPhysicsGetSteamEngineFireboxPlateThickness(vehicle),
                                  TNIPhysicsGetSteamEngineFireboxThermalConductivity(vehicle),
                                  TNIPhysicsGetSteamEngineFuelEnergy(vehicle),
                                  TNIPhysicsGetSteamEngineFuelSpecificHeatCapacity(vehicle),
                                  TNIPhysicsGetSteamEngineMaxFireCoalMass(vehicle),
                                  TNIPhysicsGetSteamEngineStartingCoal(vehicle));

  // Boiler
  m_boiler = new Boiler(TNIPhysicsGetSteamEngineBoilerVolume(vehicle),
                        TNIPhysicsGetSteamEngineStartingWater(vehicle),
                        TNIPhysicsGetSteamEngineInitialBoilerTemperature(vehicle),
                        TNIPhysicsGetSteamEngineStartingBoilerSteam(vehicle),
                        TNIPhysicsGetSteamEngineBoilerEfficencyIdle(vehicle),
                        TNIPhysicsGetSteamEngineBoilerEfficency(vehicle),
                        TNIPhysicsGetSteamEngineBoilerEfficencyMin(vehicle),
                        TNIPhysicsGetSteamEngineBoilerHeatLoss(vehicle));

  // Steam chest
  m_steamChest = new SteamChest(m_boiler,
                                TNIPhysicsGetSteamEngineSteamChestVolume(vehicle),
                                TNIPhysicsGetSteamEngineSuperheatingConstant(vehicle),
                                TNIPhysicsGetSteamEngineSteamChestMaxFlow(vehicle));

  // Piston manager
  m_pistonManager = new PistonManager(TNIPhysicsGetSteamEngineValveLapPercent(vehicle));

  // Driving wheel
  double wheelDiameter = TNIPhysicsGetVehicleWheelDiameter(vehicle);
  if (wheelDiameter == 0)
    wheelDiameter = 1;

  m_wheel = new DrivingWheel( TNIPhysicsGetVehicleMass(vehicle),
                              wheelDiameter,
                              TNIPhysicsGetVehicleDrivingWheelWeightRatio(vehicle),
                              TNIPhysicsGetEngineBrakeRatio(vehicle),
                              TNIPhysicsGetSteamEngineHandBrakeMaxForce(vehicle));

  // Driving wheel resistance parameters
  m_wheel->InitDavisFormulaValues((int)TNIPhysicsGetVehicleAxleCount(vehicle),
                                  TNIPhysicsGetVehicleSurfaceArea(vehicle),
                                  TNIPhysicsGetVehicleMovingFrictionCoefficient(vehicle),
                                  TNIPhysicsGetVehicleAirDragCoefficient(vehicle));
  m_wheel->SetMaximumSpeedHack((double)TNIPhysicsGetVehicleMaxSpeed(vehicle) * 1.1);

  // Pistons
  m_pistonCount = TNIPhysicsGetSteamEnginePistonAngularOffsetsCount(vehicle);
  m_pistons = new SteamPhysics::Piston*[m_pistonCount];
  for (int index = 0; index < m_pistonCount; ++index)
  {
    Piston* piston = new Piston(TNIPhysicsGetSteamEnginePistonArea(vehicle),
                                TNIPhysicsGetSteamEnginePistonVolumeMin(vehicle),
                                TNIPhysicsGetSteamEnginePistonVolumeMax(vehicle),
                                TNIPhysicsGetSteamEnginePistonAngularOffset(vehicle, index) / PI);
    m_pistons[index] = piston;
    m_pistonManager->AddPiston(piston);
    m_wheel->AddPiston(piston, TNIPhysicsGetSteamEnginePistonAngularOffset(vehicle, index));
  }

  // Safety valves
  m_phySafetyValveLowPressure = new SafetyValve(TNIPhysicsGetSteamEngineSafetyValveLowPressure(vehicle) * 1000.0,
                                                TNIPhysicsGetSteamEngineSafetyValveLowFlow(vehicle));
  m_phySafetyValveHighPressure = new SafetyValve( TNIPhysicsGetSteamEngineSafetyValveHighPressure(vehicle) * 1000.0,
                                                  TNIPhysicsGetSteamEngineSafetyValveHighFlow(vehicle));
  m_phyDerailedPressureRelease = new SafetyValve(101325.0, 30.0);


  // Set as much other initial state as possible. (We may also get a load
  // call, but it isn't guaranteed.)
  m_wheel->AdjustMomentum(TNIPhysicsGetVehicleWheelMomentum(vehicle));
  m_wheel->SetHasWheelslip(TNIPhysicsGetVehicleWheelslip(vehicle) > 0);
}


//=============================================================================
// Name: ~TNIPhysicsVehicleStateSteam
// Desc: Deletes the various steam engine components for this vehicle. Deletion
//       of this state structure will occur when the vehicle is unregistered
//       from the plugin, either as a result of vehicle deletion, or when
//       switching it over to a different plugin.
//=============================================================================
TNIPhysicsVehicleStateSteam::~TNIPhysicsVehicleStateSteam(void)
{
  delete m_phyDerailedPressureRelease;
  delete m_phySafetyValveHighPressure;
  delete m_phySafetyValveLowPressure;

  delete m_wheel;
  delete m_pistonManager;
  for (int i = 0; i < m_pistonCount; ++i)
    delete m_pistons[i];
  delete[] m_pistons;
  delete m_steamChest;
  delete m_boiler;
  delete m_firebox;
}


//=============================================================================
// Name: SaveVehicleState
// Desc: Saves all the state necessary to support later restoration of this
//       structure. This is used during Trainz Route/Session save, and when
//       switching a vehicle between physics plugins. This must save any and
//       all internal state which is required between frames. State save always
//       uses the latest version (TNIPhysicsContext::kSaveFormatVersion).
// Parm: io_data - (IN, OUT) A TNISoup to save our internal data to.
//=============================================================================
void TNIPhysicsVehicleStateSteam::SaveVehicleState(TNIRef<TNISoup>& io_data)
{
  TNIPhysicsVehicleState::SaveVehicleState(io_data);

  m_firebox->Save(io_data);
  m_boiler->Save(io_data);
  m_steamChest->Save(io_data);
  for (int i = 0; i < m_pistonCount; ++i)
    m_pistons[i]->Save(io_data, i);
  m_wheel->Save(io_data);

  m_phySafetyValveLowPressure->Save(io_data, g_strings->lblSafetyValveLow);
  m_phySafetyValveHighPressure->Save(io_data, g_strings->lblSafetyValveHigh);

  SoupSetFloat(io_data, g_strings->lblWaterConsumed, m_waterConsumed);
  SoupSetFloat(io_data, g_strings->lblCoalTimer, m_coalShovelTimer);
}


//=============================================================================
// Name: LoadVehicleState
// Desc: Restores internal state using the soup passed, which is assumed to
//       have been created from an earlier call to the above save function, or
//       something which generates the same data (in another 3rd party plugin,
//       for example). Missing variables will be initialised to their defaults.
// Parm: data - (IN, OUT) The TNISoup to load our internal data from.
// Parm: dataVersion - The data version at which the passed soup was written.
//=============================================================================
void TNIPhysicsVehicleStateSteam::LoadVehicleState(const TNIRef<const TNISoup>& data, int dataVersion)
{
  TNIPhysicsVehicleState::LoadVehicleState(data, dataVersion);

  m_firebox->Load(data, dataVersion);
  m_boiler->Load(data, dataVersion);
  m_steamChest->Load(data, dataVersion);
  for (int i = 0; i < m_pistonCount; ++i)
    m_pistons[i]->Load(data, i, dataVersion);
  m_wheel->Load(data, dataVersion);

  m_phySafetyValveLowPressure->Load(data, g_strings->lblSafetyValveLow, dataVersion);
  m_phySafetyValveHighPressure->Load(data, g_strings->lblSafetyValveHigh, dataVersion);

  m_waterConsumed = SoupGetFloat(data, g_strings->lblWaterConsumed, m_waterConsumed);
  m_coalShovelTimer = (float)SoupGetFloat(data, g_strings->lblCoalTimer, m_coalShovelTimer);
}


//=============================================================================
// Name: UpdateEnginePhysics
// Desc: Updates the internal engine physics for the time interval passed. In
//       essence this means burning fuel to produce heat energy, transferring
//       that energy to the boiler, using it to boil water into steam,
//       transferring that steam to the steam chest, and then (potentially)
//       allowing that steam into the pistons to drive the wheels.
// Note: This function is called for both DCC and realistic mode trains, but in
//       DCC mode the engine force output isn't used to move the train, just to
//       update the PFX and wheel animations, consume fuel, etc.
// Parm: dt - The time interval being applied in this call, in seconds.
// Parm: brakeData - A VehicleBrakeData struct constructed from the tni vehicle
//       that is being updated.
// Parm: io_resultsStream - (IN, OUT) The stream to write update commands to.
//=============================================================================
void TNIPhysicsVehicleStateSteam::UpdateEnginePhysics(float dt, VehicleBrakeData* brakeData, TNIRef<TNIStream>& io_resultsStream)
{
  int8_t physicsMode = TNIPhysicsGetVehiclePhysicsModelSetting(m_vehicle.c_obj());
  double curSpeed = fabs(TNIPhysicsGetVehicleVelocity(m_vehicle.c_obj()));
  bool bIsDerailed = TNIPhysicsGetVehicleIsDerailed(m_vehicle.c_obj());

  // Grab the cutoff and regulator settings (note: Trainz code will automate these in DCC)
  double cutoffSetting = bIsDerailed ? 0 : TNIPhysicsGetVehicleSteamCutoffSetting(m_vehicle.c_obj());
  double regulatorSetting = bIsDerailed ? 0 : TNIPhysicsGetVehicleSteamRegulatorSetting(m_vehicle.c_obj());

  m_steamChest->SetRegulatorSetting(regulatorSetting);

  double draftingSpeed;
  if (physicsMode == TNIPhysics_TrainPhysicsDCC)
    draftingSpeed = curSpeed;
  else
    draftingSpeed = fabs(m_wheel->GetWheelVelocity());


  // Update the firebox, burning fuel to create heat energy.
  m_firebox->SetBlowerSetting(TNIPhysicsGetVehicleBlowerSetting(m_vehicle.c_obj()));
  m_firebox->Update(dt, draftingSpeed, fabs(cutoffSetting),
                    m_steamChest->GetPressure(), m_boiler->GetBoilerPressure(), m_boiler->GetBoilerWaterTemp());

  // Transfer all the firebox energy into the boiler.
  m_boiler->AddEnergy(m_firebox->UseAllEnergy());

  // Update the boiler, transforming water (if there is any) into steam.
  m_boiler->Update(dt, m_firebox->GetBlowerSetting(), m_firebox->GetBlowerSteamDemand(), m_firebox->GetBurnRateFactor());

  // Release excees boiler pressure via the safety valves.
  m_phySafetyValveLowPressure->Update(m_boiler, dt);
  m_phySafetyValveHighPressure->Update(m_boiler, dt);

  if (bIsDerailed || (m_boiler->GetBoilerWaterVolume() * 1000.0) < TNIPhysicsGetSteamEngineBoilerFusiblePlugVolume(m_vehicle.c_obj()))
    m_phyDerailedPressureRelease->Update(m_boiler, dt);


  // Update the steam chest, which will pull steam from the boiler.
  m_steamChest->Update(m_boiler, dt);

  // Update the piston manager, which uses the cutoff setting to allow steam
  // from the steam chest into the pistons.
  m_pistonManager->Update(m_steamChest, dt, cutoffSetting);

  // Calculate the main stack PFX value.
  double chimneySteamExhaust =  (m_steamChest->GetPressure() * cutoffSetting) /
                                (TNIPhysicsGetSteamEngineSafetyValveLowPressure(m_vehicle.c_obj()) * 1000.0);

  // Update control state on the driving wheel.
  m_wheel->SetHandBrakeSetting((double)TNIPhysicsGetVehicleHandBrakeSetting(m_vehicle.c_obj()));
  if (TNIPhysicsGetVehicleSandingSetting(m_vehicle.c_obj()) > 0.5f)
    m_wheel->SetSandingFactor(TNIPhysicsGetVehicleSandingTractionMultiplier(m_vehicle.c_obj()));
  else
    m_wheel->SetSandingFactor(TNIPhysicsGetVehicleScriptTractionMultiplier(m_vehicle.c_obj()));

  m_wheel->SetResistanceInfo(TNIPhysicsGetVehicleTrackGradient(m_vehicle.c_obj()),
                             TNIPhysicsGetVehicleTrackCurveAngle(m_vehicle.c_obj()),
                             DetermineTractiveEffortFromWeight(dt),
                             TNIPhysicsGetVehicleWheelslipTractionMultiplier(m_vehicle.c_obj()),
                             TNIPhysicsGetVehicleWheelslipMomentumMultiplier(m_vehicle.c_obj()));

  m_wheel->SetBrakePressurePSI(GetBrakeCylinderPressurePSI(brakeData));

  // Apply the piston forces to the wheel, calculate wheelslip, etc.
  m_wheel->Update(dt);


  // Check the water injector settings (note: Trainz code will automate these
  // in DCC or if the automatic fireman is enabled).
  const double injectorSetting1 = TNIPhysicsGetVehicleInjector1Setting(m_vehicle.c_obj());
  const double injectorSetting2 = TNIPhysicsGetVehicleInjector2Setting(m_vehicle.c_obj());
  const double waterAvailable = (double)TNIPhysicsGetVehicleWaterAvailable(m_vehicle.c_obj());

  if (injectorSetting1 + injectorSetting2 > 0 && waterAvailable > 0)
  {
    // Add water to the boiler (if there's space).
    double waterRatio = m_boiler->GetBoilerWaterVolume() / m_boiler->GetBoilerVolume();
    if (waterRatio < 0.99)
    {
      // Accumulate water to add to the boiler.
      double addAmount = (TNIPhysicsGetSteamEngineWaterInjectorRate(m_vehicle.c_obj()) * injectorSetting1 +
                          TNIPhysicsGetSteamEngineWaterInjectorRate2(m_vehicle.c_obj()) * injectorSetting2) * dt;
      if (addAmount > waterAvailable)
        addAmount = waterAvailable;

      // Attempt to add the water.
      m_waterConsumed += m_boiler->AddWater(addAmount, 350);
      if (m_waterConsumed > 1)
      {
        double productConsumed = floor(m_waterConsumed);
        m_waterConsumed -= productConsumed;

        TNIStreamWriteInt16(io_resultsStream, TNIP_ConsumeWater);
        TNIStreamWriteInt32(io_resultsStream, (int)productConsumed);
      }
    }
  } // if (injectorSetting1 + injectorSetting2 > 0)


  // Check the coal rate setting (note: Trainz code will automate this in DCC
  // or if the automatic fireman is enabled).
  float coalRateSetting = TNIPhysicsGetVehicleCoalRateSetting(m_vehicle.c_obj());
  if (coalRateSetting > 0.01f)
  {
    if (m_coalShovelTimer > 0)
      m_coalShovelTimer -= dt;

    // Add coal to the firebox (if there's space)
    if (m_firebox->GetPercentFull() < 1.0 && m_coalShovelTimer <= 0)
    {
      int coalShovelProduct = (int)(TNIPhysicsGetSteamEngineCoalShovelMass(m_vehicle.c_obj()) / 0.860f);  // 0.860 = coal density
      coalShovelProduct = std::min(std::max(1, coalShovelProduct), (int)TNIPhysicsGetVehicleCoalAvailable(m_vehicle.c_obj()));

      if (coalShovelProduct > 0)
      {
        m_firebox->AddCoal(coalShovelProduct * 0.860);
        TNIStreamWriteInt16(io_resultsStream, TNIP_ConsumeCoal);
        TNIStreamWriteInt32(io_resultsStream, coalShovelProduct);

        m_coalShovelTimer = 15.f + (float)((1.f - coalRateSetting) * 285.f);
      }

    }

  } // if (coalRateSetting > 0.01f)


  if (physicsMode != TNIPhysics_TrainPhysicsDCC)
  {
    // Set the new wheel momentum
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelMomentum);
    TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetForwardMomentum());
  }

  // Update the loco mass due to any transferred water and coal. Note that we
  // used to clear this value in DCC mode, but this can place undue stress on
  // the couplers when switching between control modes.
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetExtraEngineMass);
  TNIStreamWriteFloat64(io_resultsStream, m_boiler->GetBoilerWaterMass() + m_firebox->GetCoal());

  // Update UI display values
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetCoalLevel);
  TNIStreamWriteFloat64(io_resultsStream, m_firebox->GetPercentFull());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWaterLevel);
  TNIStreamWriteFloat64(io_resultsStream, GetBoilerWaterLevel());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetBoilerPressure);
  TNIStreamWriteFloat64(io_resultsStream, m_boiler->GetBoilerPressure());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetSteamChestPressure);
  TNIStreamWriteFloat64(io_resultsStream, m_steamChest->GetPressure());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetThrashing);
  TNIStreamWriteFloat64(io_resultsStream, m_firebox->GetThrashingFactor());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelAngularVelocity);
  TNIStreamWriteFloat(io_resultsStream, (float)m_wheel->GetAngularVelocity());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelAnimationData);
  TNIStreamWriteFloat(io_resultsStream, (float)m_wheel->GetWheelRotation());
  TNIStreamWriteFloat(io_resultsStream, (float)m_wheel->GetTrainVelocity());

  TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineForce);
  TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetEngineForce());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetAppliedEngineForce);
  TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetAppliedEngineForce());
  if (physicsMode != TNIPhysics_TrainPhysicsDCC)
  {
    // Never use wheelslip in DCC
    TNIStreamWriteInt16(io_resultsStream, TNIP_SetWheelslipForce);
    TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetWheelslipForce());
  }
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetBrakingForce);
  TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetBrakingForce());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetResistanceForce);
  TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetResistanceForce());

  TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineParam);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblFireTemperature.c_obj());
  TNIStreamWriteFloat64(io_resultsStream, m_firebox->GetFireTemperature());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineParam);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblCoalMass.c_obj());
  TNIStreamWriteFloat64(io_resultsStream, m_firebox->GetCoal());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetEngineParam);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblSteamPistonCycle.c_obj());
  TNIStreamWriteFloat64(io_resultsStream, m_wheel->GetWheelRotation());

  // Update steam PFX variables
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXVariableValue);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblPFXSteamOutletStack.c_obj());
  TNIStreamWriteFloat(io_resultsStream, (float)chimneySteamExhaust);
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXVariableValue);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblPFXSteamOutletLowPressureValve.c_obj());
  TNIStreamWriteFloat(io_resultsStream, (float)m_phySafetyValveLowPressure->GetCurrentFlowRate());
  TNIStreamWriteInt16(io_resultsStream, TNIP_SetPFXVariableValue);
  TNIStreamWriteTNIObjectReference(io_resultsStream, g_strings->lblPFXSteamOutletHighPressureValve.c_obj());
  TNIStreamWriteFloat(io_resultsStream, (float)m_phySafetyValveHighPressure->GetCurrentFlowRate());

}


//=============================================================================
// Name: GetForwardMomentum
// Desc: Returns the current forward momentum of this traincar. This is the
//       value that's been calculated and stored on this class, and may not
//       exactly match the value on the linked TNIPhysicsVehicle.
// Retn: double - The current forward momentum of this traincar, in kg.m/s.
//=============================================================================
double TNIPhysicsVehicleStateSteam::GetForwardMomentum(void) const
{
  return m_wheel->GetForwardMomentum();
}


//=============================================================================
// Name: UpdateMass
// Desc: Called by Trainz to apply external updates to overall vehicle mass
//       (caused by fuel consumption or other product transfer, for example).
// Parm: newVehicleMass - The new mass of this traincar and it's load, in kg.
//=============================================================================
void TNIPhysicsVehicleStateSteam::UpdateMass(double newVehicleMass)
{
  m_wheel->UpdateMass(newVehicleMass);
}


//=============================================================================
// Name: AdjustMomentum
// Desc: Called by Trainz to adjust the momentum of this train (due to coupler
//       forces, for example, multiplayer sync adjustsments, etc).
// Parm: newMomentum - The new momentum of this traincar, in kg.m/s.
//=============================================================================
void TNIPhysicsVehicleStateSteam::AdjustMomentum(double newMomentum)
{
  m_wheel->AdjustMomentum(newMomentum);
}


//=============================================================================
// Name: FinaliseFrame
// Desc: Called by Trainz to notify that all physics updates for the current
//       frame are complete, allowing us to sync the results.
// Parm: finalMomentum - The final frame momentum of this traincar, in kg.m/s.
//       This includes final calculations of all engine, resistance, and
//       coupler forces over the entire train.
// Parm: dt - The time interval being applied in this call, in seconds.
//=============================================================================
void TNIPhysicsVehicleStateSteam::FinaliseFrame(double finalMomentum, float dt)
{
  // Synchronise the final momentum.
  m_wheel->AdjustMomentum(finalMomentum);

  // We also use this notification to update the wheel animation.
  m_wheel->UpdatePosition(dt);
}


//=============================================================================
// Name: AddCoalToFire
// Desc: Called by Trainz to add fuel to the engine, usually as a result of the
//       player using the "shovel coal" command.
// Parm: coalVolume - The volume of coal to add. This is in "product units",
//       so doesn't have an exact units definition, but we assume a single
//       unit to have a mass of 0.860kg.
//=============================================================================
void TNIPhysicsVehicleStateSteam::AddCoalToFire(double coalVolume)
{
  double coalMass = coalVolume * 0.860;
  m_firebox->AddCoal(coalMass);
}


//=============================================================================
// Name: DetermineTractiveEffortFromWeight
// Desc: Determines how much traction this traincar has based on slope, mass
//       and wheel adhesion. This value diminishes as track grade increases.
// Parm: dt - The time interval being applied in this call, in seconds.
// Retn: double - Vehicle tractive force (in Newtons).
//=============================================================================
double TNIPhysicsVehicleStateSteam::DetermineTractiveEffortFromWeight(float dt) const
{
  // TE = Wd * (1-G/100) * A
  //
  // Wd = Weight on the driving wheels
  // G = % grade (upgrade +, downgrade -)
  // A = Adhesion factor

  double mass = TNIPhysicsGetVehicleMass(m_vehicle.c_obj());
  double grade = fabs(TNIPhysicsGetVehicleTrackGradient(m_vehicle.c_obj()));

  double adhesion = 0.23; // 23% adhesion (from Eric - divide weight by 4.2)

  return mass * (1.0 - grade) * adhesion * 9.81;
}


//=============================================================================
// Name: GetBrakeCylinderPressure
// Desc: Returns the brake cylinder pressure for applying braking force to this
//       traincar. This will return whichever is greatest out of the train
//       brake, independent brake, or script brake pressure.
// Parm: brakeData - The current brake data for this traincar.
// Retn: double - The current braking pressure for this traincar, in PSI.
//=============================================================================
double TNIPhysicsVehicleStateSteam::GetBrakeCylinderPressurePSI(VehicleBrakeData* brakeData) const
{
  const double abcP = brakeData->autoBrakeCylinder.GetPressure();
  const double ibcP = brakeData->independentBrakeCylinder.GetPressure();
  const double sbP = TNIPhysicsGetVehicleScriptBrakePressure(m_vehicle.c_obj());

  double maxPressure = UnitConversion::gm3_PSI(std::max(sbP, std::max(abcP, ibcP)));
  return maxPressure;
}


//=============================================================================
// Name: GetBoilerWaterLevel
// Desc: Returns the level (0-1) to show on the water gauge. It's important to
//       note that 0 is not 'empty', just 0 on the gauge (ie, the 'minimum').
//       This is used both for the water display in the HUD/cab, and by the
//       injector update code in UpdateEnginePhysics().
// Retn: double - As above, the water gauge position from 0 (min) to 1 (full).
//=============================================================================
double TNIPhysicsVehicleStateSteam::GetBoilerWaterLevel(void) const
{
  // Note that the boiler volume on the engine spec is in cubic metres, but the
  // water gauge min/max values are in litres.
  const double specVol = TNIPhysicsGetSteamEngineBoilerVolume(m_vehicle.c_obj()) * 1000.0;
  const double gaugeMinVol = TNIPhysicsGetSteamEngineBoilerVolumeMin(m_vehicle.c_obj()) / specVol;
  const double gaugeMaxVol = TNIPhysicsGetSteamEngineBoilerVolumeMax(m_vehicle.c_obj()) / specVol;

  // Calculate the percentage of the boiler that is occupied by water.
  double waterRatio = m_boiler->GetBoilerWaterVolume() / m_boiler->GetBoilerVolume();

  waterRatio = (waterRatio - gaugeMinVol) / (gaugeMaxVol - gaugeMinVol);

  return std::max(0.0, std::min(waterRatio, 1.0));
}


