//=============================================================================
// File: TrainPhysicsSteam.h
// Desc: Declares the main engine force update function for realistic mode
//       steam trains, and TNIPhysicsVehicleStateSteam, which is used to track
//       between-frame state for steam engines (in both dcc and realistic).
//=============================================================================
#ifndef _TrainPhysicsSteam_H_
#define _TrainPhysicsSteam_H_
#pragma once


#include "TNIPhysicsVehicleState.h"
#include <vector>

class TNIPhysicsVehicle;
class TNIPhysicsVehicleStateSteam;
class VehicleBrakeData;
class TNIStream;
namespace SteamPhysics
{
  class FireBox;
  class Boiler;
  class SteamChest;
  class PistonManager;
  class Piston;
  class DrivingWheel;
  class SafetyValve;
}; // SteamPhysics



double UpdateEngineForceSteam(TNIPhysicsVehicleStateSteam* vehicle, float dt, TNIRef<TNIStream>& io_resultsStream);



//=============================================================================
// Name: TNIPhysicsVehicleStateSteam
// Desc: The TNIPhysicsVehicleState override used for steam trains. We track a
//       *lot* of data between frames for steam engines, and this class is
//       responsible for allocating, managing, saving and reloading it.
//=============================================================================
class TNIPhysicsVehicleStateSteam : public TNIPhysicsVehicleState
{
public:

  TNIPhysicsVehicleStateSteam(const TNIPhysicsVehicle* vehicle);
  virtual ~TNIPhysicsVehicleStateSteam(void);

  virtual void SaveVehicleState(TNIRef<TNISoup>& io_data) override;
  virtual void LoadVehicleState(const TNIRef<const TNISoup>& data, int dataVersion) override;

  virtual void UpdateEnginePhysics(float dt, VehicleBrakeData* brakeData, TNIRef<TNIStream>& io_resultsStream) override;


  double GetForwardMomentum() const;
  void UpdateMass(double newVehicleMass);

  void AdjustMomentum(double newMomentum);
  void FinaliseFrame(double finalMomentum, float dt);
  void AddCoalToFire(double coalVolume);

  double DetermineTractiveEffortFromWeight(float dt) const;

  double GetBrakeCylinderPressurePSI(VehicleBrakeData* brakeData) const;
  double GetBoilerWaterLevel(void) const;



private:

  // Pointers to our various steam engine components. These track fine grained
  // physics state between frames and are allocated and destroyed by this class
  // at construction/destruction time. Each class has a save and load function
  // which allows this state to be saved and restored between runs or when
  // switching plugins on a vehicle. See the relevant class headers for more
  // info on any of the specific objects.
  SteamPhysics::FireBox*          m_firebox = nullptr;
  SteamPhysics::Boiler*           m_boiler = nullptr;
  SteamPhysics::SteamChest*       m_steamChest = nullptr;
  SteamPhysics::Piston**          m_pistons = nullptr;
  int                             m_pistonCount = 0;
  SteamPhysics::PistonManager*    m_pistonManager = nullptr;
  SteamPhysics::DrivingWheel*     m_wheel = nullptr;

  SteamPhysics::SafetyValve*      m_phySafetyValveLowPressure = nullptr;
  SteamPhysics::SafetyValve*      m_phySafetyValveHighPressure = nullptr;
  SteamPhysics::SafetyValve*      m_phyDerailedPressureRelease = nullptr;


  double      m_waterConsumed = 0;    // Fractional water product consumed.
  float       m_coalShovelTimer = 0;  // Time until next automatic coal shovel.


};


#endif // _TrainPhysicsSteam_H_

