//=============================================================================
// File: Conversion.cpp
// Desc: Contains constants and static functions related to unit conversions.
//=============================================================================
#include "UnitConversion.h"
#include "TNIPhysicsGlobals.h"



//=============================================================================
// Name: pa_psi
// Desc: Converts from pascal (pa) to PSI (pounds per square inch).
//=============================================================================
double UnitConversion::pa_psi(double pa)
{
  return pa / 6894.7;
}


//=============================================================================
// Name: pa_bar
// Desc: Converts from pascal (pa) to bar.
//=============================================================================
double UnitConversion::pa_bar(double pa)
{
  return pa / 100000;
}


//=============================================================================
// Name: PSI_gm3
// Desc: Converts from PSI (pounds per square inch) to gm3 (grams per metre
//       cubed).
// Note: This function takes psi relative to the earths atmosphere. For
//       pressures relative to a vacuum (i.e. absolute) see APSI_gm3.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::PSI_gm3(double psi)
{
  // 1 PSI  =  0.0703 kg/cm3  =  70.3 g/cm3  =  0.0000703 g/m3
  return psi * 0.0000703 + kAtmosphericPressureGM3;
}


//=============================================================================
// Name: gm3_PSI
// Desc: Converts from gm3 (grams per metre cubed) to PSI (pounds per square
//       inch).
// Note: This function returns pressure relative to the earths atmosphere. For
//       pressures relative to a vacuum (i.e. absolute) see gm3_APSI.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::gm3_PSI(double gm3)
{
  return (gm3 - kAtmosphericPressureGM3) / 0.0000703;
}


//=============================================================================
// Name: APSI_gm3
// Desc: Converts from absolute PSI (pounds per square inch) to gm3 (grams per
//       metre cubed).
// Note: This function takes PSI relative to a vacuum. For pressures relative
//       to the earths atmosphere see PSI_gm3.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::APSI_gm3(double psi)
{
  // 1 PSI = 0.0703 kg/cm3 = 70.3 g/cm3 = 0.0000703
  return psi * 0.0000703;
}


//=============================================================================
// Name: gm3_APSI
// Desc: Converts from gm3 (grams per metre cubed) to absolute PSI (pounds per
//       square inch).
// Note: This function returns pressure relative to a vacuum. For pressures
//       relative to the earths atmosphere see gm3_PSI.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::gm3_APSI(double gm3)
{
  return gm3 / 0.0000703;
}


//=============================================================================
// Name: kpa_gm3
// Desc: Converts from kPa (kilopascals) to gm3 (grams per metre cubed).
// Note: This function assumes kPa relative to the earths atmosphere. For
//       pressures relative to a vacuum (i.e. absolute) see akpa_gm3.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::kpa_gm3(double kPa)
{
  // http://www.control.co.kr/dic/press_conv.htm
  // gm3 = kPa * 0.1450 * 0.0000703
  return kPa * 0.145 * 0.0000703 + kAtmosphericPressureGM3;
}


//=============================================================================
// Name: gm3_kpa
// Desc: Converts from gm3 (grams per metre cubed) to kPa (kilopascals).
// Note: This function returns pressure relative to the earths atmosphere. For
//       pressures relative to a vacuum (i.e. absolute) see gm3_akpa.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::gm3_kpa(double gm3)
{
  return (gm3 - kAtmosphericPressureGM3) / (0.145 * 0.0000703);
}


//=============================================================================
// Name: akpa_gm3
// Desc: Converts from absolute kPa (kilopascals) to gm3 (grams per metre
//       cubed).
// Note: This function takes kPa relative to a vacuum. For pressures relative
//       to the earths atmosphere see kpa_gm3.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::akpa_gm3(double akPa)
{
  return akPa * 0.145 * 0.0000703;
}


//=============================================================================
// Name: gm3_akpa
// Desc: Converts from gm3 (grams per metre cubed) to absolute kpa
//       (kilopascals).
// Note: This function returns pressure relative to a vacuum. For pressures
//       relative to the earths atmosphere see gm3_kpa.
// Note: Technically, gm3 is a measure of density, not pressure, and this
//       conversion relies on the principals of ideal gas law.
//       https://en.wikipedia.org/wiki/Ideal_gas_law
//=============================================================================
double UnitConversion::gm3_akpa(double gm3)
{
  return gm3 / (0.145 * 0.0000703);
}


//=============================================================================
// Name: hg_kpa
// Desc: Converts from torr (aka mmHg, or millimeter of mercury) to kPa.
//=============================================================================
double UnitConversion::hg_kpa(double hg)
{
  // http://www.control.co.kr/dic/press_conv.htm
  return hg * 0.13332;
}


//=============================================================================
// Name: in3_m3
// Desc: Converts from cubic inches to cubic metres.
//=============================================================================
double UnitConversion::in3_m3(double in3)
{
  // http://www.connel.com/cgi-bin/convert.pl
  return in3 * 0.0000163871;
}


//=============================================================================
// Name: f3_m3
// Desc: Converts from cubic feet to cubic metres.
//=============================================================================
double UnitConversion::f3_m3(double f3)
{
  // http://www.connel.com/cgi-bin/convert.pl
  return f3 * 0.028317;
}


//=============================================================================
// Name: l_m3
// Desc: Converts from litres to cubic metres.
//=============================================================================
double UnitConversion::l_m3(double l)
{
  // http://www.connel.com/cgi-bin/convert.pl
  return l * 0.001;
}


//=============================================================================
// Name: kg_ton
// Desc: Converts from kilograms to US ton.
//=============================================================================
double UnitConversion::kg_ton(double kg)
{
  // http://quest.arc.nasa.gov/space/frontiers/activities/aeronautics/hints.html
  return kg / 907.2;
}


//=============================================================================
// Name: ton_kg
// Desc: Converts from US ton to kilograms.
//=============================================================================
double UnitConversion::ton_kg(double ton)
{
  // http://quest.arc.nasa.gov/space/frontiers/activities/aeronautics/hints.html
  return ton * 907.2;
}


//=============================================================================
// Name: mph_mps
// Desc: Converts from miles per hour (mph) to metres per second (mps).
//=============================================================================
double UnitConversion::mph_mps(double mph)
{
  // http://www.srh.noaa.gov/wgrfc/convert.html
  return mph * 0.4740;
}


//=============================================================================
// Name: mps_mph
// Desc: Converts from metres per second (mps) to miles per hour (mph).
//=============================================================================
double UnitConversion::mps_mph(double mps)
{
  // http://www.srh.noaa.gov/wgrfc/convert.html
  return mps * 2.237;
}


//=============================================================================
// Name: kmh_mps
// Desc: Converts from kilometres per hour (kmh) to metres per second (mps).
//=============================================================================
double UnitConversion::kmh_mps(double kmh)
{
  return kmh / 3.6;
}


//=============================================================================
// Name: mps_kmh
// Desc: Converts from metres per second (mps) to kilometres per hour (kmh).
//=============================================================================
double UnitConversion::mps_kmh(double mps)
{
  return mps * 3.6;
}


//=============================================================================
// Name: lbf_kn
// Desc: Converts from pounds force (lbf) to kilonewtons (kN). Multiply the
//       return value by 1000 to convert it to newtons.
//=============================================================================
double UnitConversion::lbf_kn(double lbf)
{
  return lbf * 0.00444822162;
}


//=============================================================================
// Name: kn_lbf
// Desc: Converts from kilonewtons (kN) to pounds force (lbf).
//=============================================================================
double UnitConversion::kn_lbf(double kN)
{
  return kN * 224.808943;
}