//=============================================================================
// File: VehicleBrakeData.cpp
// Desc: Defines VehicleBrakeData, a small helper class to reduce code
//       duplication around brake system updates.
//=============================================================================
#include "VehicleBrakeData.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"



//=============================================================================
// Name: VehicleBrakeData
// Desc: Initialises this structure from the engine spec of the vehicle passed.
// Parm: veh - The vehicle to initialise this structure from.
//=============================================================================
VehicleBrakeData::VehicleBrakeData(const TNIPhysicsVehicle* veh)
  : thisVeh(veh)
  , equaliser(TNIPhysicsGetVehicleVolume_Equaliser(veh))
  , mainReservoir(TNIPhysicsGetVehicleVolume_MainReservoir(veh))
  , auxReservoir(TNIPhysicsGetVehicleVolume_AuxReservoir(veh))
  , autoBrakeCylinder(TNIPhysicsGetVehicleVolume_AutoBrakeCylinder(veh))
  , independentBrakeCylinder(TNIPhysicsGetVehicleVolume_IndependentBrakeCylinder(veh))
  #if ENABLE_VACUUM_BRAKES
  , vacuumReservoir(TNIPhysicsGetVehicleVacuumBrakeReservoirVolume(vehicle));
  , vacuumBrakeCylinder(TNIPhysicsGetVehicleVacuumBrakeCylinderVolume(vehicle));
  #endif // ENABLE_VACUUM_BRAKES
{
  equaliser.SetPressure(TNIPhysicsGetVehicleEqualiserPressure(veh));
  mainReservoir.SetPressure(TNIPhysicsGetVehicleMainReservoirPressure(veh));
  auxReservoir.SetPressure(TNIPhysicsGetVehicleAuxReservoirPressure(veh));
  autoBrakeCylinder.SetPressure(TNIPhysicsGetVehicleAutoBrakeCylinderPressure(veh));
  independentBrakeCylinder.SetPressure(TNIPhysicsGetVehicleIndependentBrakeCylinderPressure(veh));
  #if ENABLE_VACUUM_BRAKES
    vacuumReservoir.SetPressure(TNIPhysicsGetVehicleVacuumBrakeReservoirPressure(vehicle));
    vacuumBrakeCylinder.SetPressure(TNIPhysicsGetVehicleVacuumBrakeCylinderPressure(vehicle));
  #endif // ENABLE_VACUUM_BRAKES
}


//=============================================================================
// Name: WriteUpdateCommandsToStream
// Desc: Called at the end of a physics update to write update commands for the
//       values stored on this structure.
// Parm: io_resultsStream - (IN, OUT) A stream to write the update commands to,
//       so that they can be passed on to Trainz.
//=============================================================================
void VehicleBrakeData::WriteUpdateCommandsToStream(TNIRef<TNIStream>& io_resultsStream)
{
  // TODO: Write update command only if changed?
  #define WRITE_PRESSURE(command, pneumatic)                                  \
    if (pneumatic.Exists())                                                   \
    {                                                                         \
      TNIStreamWriteInt16(io_resultsStream, command);                         \
      TNIStreamWriteFloat64(io_resultsStream, pneumatic.GetPressure());       \
    }

  WRITE_PRESSURE(TNIP_SetEqualiserPressure, equaliser);
  WRITE_PRESSURE(TNIP_SetMainReservoirPressure, mainReservoir);
  WRITE_PRESSURE(TNIP_SetAuxReservoirPressure, auxReservoir);
  WRITE_PRESSURE(TNIP_SetAutoBrakeCylinderPressure, autoBrakeCylinder);
  WRITE_PRESSURE(TNIP_SetIndependentBrakeCylinderPressure, independentBrakeCylinder);
  #if ENABLE_VACUUM_BRAKES
    WRITE_PRESSURE(TNIP_SetVacuumReservoirPressure, vacuumBrakeBrakeReservoir);
    WRITE_PRESSURE(TNIP_SetVacuumBrakeCylinderPressure, vacuumBrakeCylinder);
  #endif // ENABLE_VACUUM_BRAKES
}



