//=============================================================================
// File: VehicleBrakePipeData.cpp
// Desc: Defines VehicleBrakePipeData, a small helper class to reduce code
//       duplication around brake system updates.
//=============================================================================
#include "VehicleBrakePipeData.h"
#include "TNIStream.hpp"
#include "TNIPhysics.hpp"



//=============================================================================
// Name: VehicleBrakePipeData
// Desc: Initialises this structure from the engine spec of the vehicle passed.
// Parm: brakePipeSpecPressure - The brake pipe pressure when fully charged, as
//       per the engine asset spec. This is used to determine a conversion rate
//       when coupling traincars with different operating brake pressures.
// Parm: veh - The vehicle to initialise this structure from.
//=============================================================================
VehicleBrakePipeData::VehicleBrakePipeData(float brakePipeSpecPressure, const TNIPhysicsVehicle* veh)
  : thisVeh(veh)
  , pressureRatio(brakePipeSpecPressure / TNIPhysicsGetVehiclePressure_BrakePipe(veh))
  , trainBrakePipe(TNIPhysicsGetVehicleVolume_TrainBrakePipe(veh))
  , no3Pipe(TNIPhysicsGetVehicleVolume_No3Pipe(veh))
  , no4Pipe(TNIPhysicsGetVehicleVolume_No4Pipe(veh))
  , epReservoirPipe(TNIPhysicsGetVehicleVolume_EpReservoirPipe(veh))
#if ENABLE_VACUUM_BRAKES
  , vacuumBrakePipe(TNIPhysicsGetVehicleVolume_VacuumBrakePipe(veh))
#endif // ENABLE_VACUUM_BRAKES
{
  trainBrakePipe.SetPressure(TNIPhysicsGetVehicleTrainBrakePipePressure(veh));
  no3Pipe.SetPressure(TNIPhysicsGetVehicleNo3PipePressure(veh));
  no4Pipe.SetPressure(TNIPhysicsGetVehicleNo4PipePressure(veh));
  epReservoirPipe.SetPressure(TNIPhysicsGetVehicleEpReservoirPipePressure(veh));
  #if ENABLE_VACUUM_BRAKES
    vacuumBrakePipe.SetPressure(TNIPhysicsGetVehicleVacuumBrakePipePressure(veh));
  #endif // ENABLE_VACUUM_BRAKES
}


//=============================================================================
// Name: VehicleBrakePipeData::WriteUpdateCommandsToStream
// Desc: Called at the end of a physics update to write update commands for the
//       values stored on this structure.
// Parm: io_resultsStream - (IN, OUT) A stream to write the update commands to,
//       so that they can be passed on to Trainz.
//=============================================================================
void VehicleBrakePipeData::WriteUpdateCommandsToStream(TNIRef<TNIStream>& io_resultsStream)
{
  // TODO: Write update command only if changed?
  #define WRITE_PRESSURE(command, pneumatic)                                  \
    if (pneumatic.Exists())                                                   \
    {                                                                         \
      TNIStreamWriteInt16(io_resultsStream, command);                         \
      TNIStreamWriteFloat64(io_resultsStream, pneumatic.GetPressure());       \
    }

  WRITE_PRESSURE(TNIP_SetTrainBrakePipePressure, trainBrakePipe);
  WRITE_PRESSURE(TNIP_SetNo3PipePressure, no3Pipe);
  WRITE_PRESSURE(TNIP_SetNo4PipePressure, no4Pipe);
  WRITE_PRESSURE(TNIP_SetEpReservoirPipePressure, epReservoirPipe);
  #if ENABLE_VACUUM_BRAKES
    WRITE_PRESSURE(TNIP_SetVacuumBrakePipePressure, vacuumBrakePipe);
  #endif // ENABLE_VACUUM_BRAKES
}


