//=============================================================================
// File: TNISample.cpp
// Desc: The TNISample library is a sample TNI plugin.
//=============================================================================
#include "TNISample.h"
#define TRAINZ_FORCE_TNI_C_API 1 // this sample demonstrates the C api, even though it's written in C++
#include "trainznativeinterface.h"
#include "TNIFunctions.h"
#include "TNILibrary.h"
#include "TNISTLAlloc.h"
#include "TNILabel.h"
#include "TNIDebug.h"
#include "TNIContext.h"
#include <mutex>


static std::mutex s_initMutex;
static unsigned int s_libraryCount = 0;
static TNILabel* s_labelSampleCall = nullptr;


static void LibraryShutdown(TNIContext* context, TNILibrary* library);
static TNIObject* LibraryCall(TNIContext* context, const TNILibrary* library, const TNILabel* functionName, const TNIObject* params);
static TNIObject* PerformSampleCall(TNIContext* context, const TNILibrary* self, const TNIObject* params);




//=============================================================================
// Name: TNIEntryPoint
// Desc: DLL Entry point, performs basic initialisation. Note that this may be
//       called once per context, which may be sequential or simultaneous. As
//       Trainz is heavily multithreaded, it is recommended that you minimise
//       the use of global variables and statics.
//=============================================================================
extern "C" TNI_ENTRYPOINT_DECL TNILibrary* TNIEntryPoint(TNIContext* context)
{
  TNILibraryInitData info;
  info.m_libraryVersion = 1;
  info.m_libraryCall = &LibraryCall;
  info.m_shutdownCall = &LibraryShutdown;

  TNILabel* libraryName = TNIAllocLabel("TNISample");
  TNILibrary* library = TNIAllocLibrary(context, libraryName, info);
  TNIRelease(libraryName);
  
  s_initMutex.lock();
  if (s_libraryCount++ == 0)
  {
    // First library. Initialise this DLL's global state.
    s_labelSampleCall = TNIAllocLabel("SampleCall");
  }
  s_initMutex.unlock();
  
  return library;
}


//=============================================================================
// Name: LibraryShutdown
// Desc: Called when Trainz is finished with a particular library instance.
//       Other library instances may still exist.
//=============================================================================
void LibraryShutdown(TNIContext* context, TNILibrary* library)
{
  s_initMutex.lock();
  if (--s_libraryCount == 0)
  {
    // No remaining libraries. Release the DLL's global state.
    TNIRelease(s_labelSampleCall);
    s_labelSampleCall = nullptr;
  }
  s_initMutex.unlock();
}


//=============================================================================
// Name: LibraryCall
// Desc: DLL Library call. Called from Trainz to ask this TNI library to do
//       something interesting. It's up to us to parse the function name and
//       parameters and decide exactly what it is that we're supposed to be
//       doing.
// Parm: context - The context in which we are being called.
// Parm: library - The library (as alloated by TNIEntryPoint) on which the call
//       is being made.
// Parm: functionName - A label indicating the function to action.
// Parm: params - The parameters to the function. Exactly what form these take
//       depends on the specific requirements of the function.
//=============================================================================
static TNIObject* LibraryCall(TNIContext* context, const TNILibrary* library, const TNILabel* functionName, const TNIObject* params)
{
  if (functionName == s_labelSampleCall)
    return PerformSampleCall(context, library, params);
  
  // Not a supported function of this library.
  return nullptr;
}


//=============================================================================
// Name: PerformSampleCall
// Desc: A sample library function, called from Trainz (TrainzScript in this
//       case) via the LibraryCall() mechanism.
// Parm: context - The context in which we are being called.
// Parm: self - The library upon which this call was placed.
// Parm: params - The parameters passed from TrainzScript. For example/debug
//       purposes, we simply convert these to a string and log it.
//=============================================================================
static TNIObject* PerformSampleCall(TNIContext* context, const TNILibrary* self, const TNIObject* params)
{
  // Log a few things to the TANE log, to demonstrate that we are alive.
  TNILog(context, "TNISample.PerformSampleCall");
  TNILogObject(context, params, 0);
  
  // Also send a script message.
  // (Note that we perform a lot of allocations and lookups in situ here which would
  // probably be better performed at library creation time. This is suitable for an
  // occasional usage but if you were to do anything frequent then you should think
  // about caching things for performance.)
  TNILabel* trainzScriptLibraryName = TNIAllocLabel("TNITrainzScript");
  TNILabel* postMessageFunctionName = TNIAllocLabel("PostMessage");
  
  TNILibrary* trainzScriptLibrary = TNIGetContextLibrary(context, trainzScriptLibraryName);
  
  if (trainzScriptLibrary)
  {
    TNILabel* messageMajor = TNIAllocLabel("TNISample");
    TNILabel* messageMinor = TNIAllocLabel("SampleMessage");
    TNIArray* params = TNIAllocArrayWith3(self, messageMajor, messageMinor);
    
    TNIRelease(TNICallLibraryFunction(trainzScriptLibrary, postMessageFunctionName, params));
    
    TNIRelease(params);
    TNIRelease(messageMinor);
    TNIRelease(messageMajor);
  }
  
  TNIRelease(postMessageFunctionName);
  TNIRelease(trainzScriptLibraryName);
  
  return nullptr;
}



