// ============================================================================
// File: TNIDebug.h
// Desc: This file provides debug helper functions for TNI. These are intended
//       for use in developing TNI dlls and should be avoided in release
//       versions.
// ============================================================================
#ifndef _TNIDEBUG_H
#define _TNIDEBUG_H
#pragma once

#include "trainznativeinterface.h"
#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif //  __cplusplus


//=============================================================================
#ifdef _DEBUG
# define USE_TNI_ASSERT 1
#else
# define USE_TNI_ASSERT 0
#endif


//=============================================================================
// Debugging assertion macros.
#if USE_TNI_ASSERT
# ifdef _WIN64
#   define TNI_ASSERT(expression) if (expression) ; else __debugbreak();
#   define TNI_ASSERT_ONCE(expression, var) do { static bool var = false; if (!var) { if (!(expression)) { var = true; __debugbreak(); } } } while (false)
# else
#   define TNI_ASSERT(expression) (void)0 // if (expression) ; else __asm int 3; -- TODO: doesn't compile.
#   define TNI_ASSERT_ONCE(expression, var) (void)0 //  -- TODO: doesn't compile.
# endif
#else
# define TNI_ASSERT(expression) (void)0
# define TNI_ASSERT_ONCE(expression, var) (void)0
#endif



// ============================================================================
// Name: TNILog
// Desc: Writes UTF8 text to the host environment's debug log.
// Parm: context - The context from which the log originates.
// Parm: textStringUTF8 - The text to output.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
void TNILog(const TNIContext* context, const char* textStringUTF8);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNILogObject
// Desc: Writes text to the host environment's debug log which best represents
//       the supplied TNIObject. The output is not intended to be machine-
//       parseable.
// Parm: context - The context from which the log originates.
// Parm: object - The TNIObject to log.
// Parm: indentLevel - The number of indentation levels at which we are
//       logging.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
void TNILogObject(const TNIContext* context, const TNIObject* object, size_t indentLevel = 0);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNILogf
// Desc: Writes UTF8 text to the host environment's debug log.
// Parm: context - The context from which the log originates.
// Parm: formatStringUTF8 - A 'printf'-style format string.
// Parm: ... - The arguments as specified by the format string.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
void TNILogf(const TNIContext* context, const char* formatStringUTF8, ...);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// Name: TNILogv
// Desc: Writes UTF8 text to the host environment's debug log.
// Parm: context - The context from which the log originates.
// Parm: formatStringUTF8 - A 'printf'-style format string.
// Parm: argv - The arguments as specified by the format string.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
void TNILogv(const TNIContext* context, const char* formatStringUTF8, va_list argv);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP






// ============================================================================
// END COMPILER OPTIONS
// ============================================================================
#ifdef __cplusplus
}
#endif // __cplusplus

#endif // _TNIDEBUG_H
