// ============================================================================
// File: TNIFunctions.h
// Desc: This C++ header provides an interface to the trainznativeinterface.dll
//       which is included in the Trainz /bin folder. This is a public 
//       interface for third-party developers to write native extensions to
//       Trainz.
//       Functions and classes in this file provide interoperability primitives
//       which allow data to be passed between Trainz and third-party DLLs.
// ============================================================================
#ifndef _TNIFunctions_H
#define _TNIFunctions_H
#pragma once

#include "trainznativeinterface.h"

//=============================================================================
//=============================================================================
#if defined(__cplusplus) && !TRAINZ_NATIVE_INTERFACE_CPP && !defined(_TNIFunctions_HPP) && !TRAINZ_FORCE_TNI_C_API
  #error "TNIFunctions.h - you should probably be including TNIFunctions.hpp instead"
#endif // ..



// ============================================================================
// COMPILER OPTIONS
//
// Despite being written in C++, this interface is considered portable and 
// third-party developers are able to access the trainznativeinterface.dll 
// from any native language which is capable of building a DLL by building an
// alternative header in that language.
// 
// Exceptions must not be thrown in DLL code.
// ============================================================================
#ifdef __cplusplus
extern "C" {
#endif // __cplusplus


// ============================================================================
// TNIOBJECT FUNCTIONS
// ============================================================================



// ============================================================================
// Name: TNIGetObjectClass
// Desc: Return the type of the specified object, as per the above enumeration
//       values.
// Parm: object - the object to query. It is safe to pass a NULL value here.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
uint32_t TNIGetObjectClass(const TNIObject* object);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP




// ============================================================================
// GENERAL FUNCTIONS
// ============================================================================

enum TNIProductRight
{
  TNI_PR_Surveyor           = 0,    // Player can use Surveyor
  TNI_PR_ContentManagment   = 1,    // Player can use content management tools
  TNI_PR_BetaVersion        = 2,    // Build is considered a beta and should validate against Planet Auran.
  TNI_PR_Driver             = 3,    // Player can use Driver.
  TNI_PR_GameplayMenu       = 4,    // Player can use gameplay menu screen.
  TNI_PR_SpeedTree          = 5,    // Player can use load the SpeedTree plugin.
};

bool TNICheckUserHasProductRight(TNIContext* context, TNIProductRight right);



// ============================================================================
// TNISTRING FUNCTIONS
// ============================================================================


// ============================================================================
// Name: TNIAllocString
// Desc: Allocates a TNIString given a zero-terminated UTF8 text buffer.
// Parm: utf8text - The UTF-8 text buffer. The contents of the buffer are
//       copied into the TNIString. The function results are undefined if this
//       parameter does not represent a valid UTF-8 encoded string.
// Retn: TNIString - The newly allocated TNIString.
// Note: You must eventually release the allocated string using TNIRelease().
// Note: If 'utf8text' is nullptr, the returned TNIString is an empty string.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIString* TNIAllocString(const char* utf8text);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetStringText
// Desc: Retrieves a pointer to the internal text buffer of the TNIString
//       object. This is a read-only reference, and remains valid only while
//       (1) you hold a reference to the TNIString, and (2) you do not modify
//       the TNIString.
// Parm: string - The TNIString object to query.
// Retn: const char* - A temporary pointer to the internal string text, in
//       utf8 encoding.
// Note: If 'string' is a nullptr, the return value is a pointer to an empty
//       string.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
const char* TNIGetStringText(const TNIObject* string);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetStringTextAndReleaseUnsafe
// Desc: Retrieves a reference to the TNIString object's internal string
//       storage but simultaneously releases a string reference. This will
//       invalidate the reference immediately if it happened to be the last
//       reference to the object, so this is only safe to use if you know for
//       sure that you have a second reference on the string. For example, this
//       can be used when extracting a string from an array, since you know
//       that the array will maintain a reference to the string even though you
//       are releasing yours.
// Parm: string - A string object which must have at least two references.
// Retn: const char* - A pointer to the internal buffer of the TNIString
//       object.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
const char* TNIGetStringTextAndReleaseUnsafe(const TNIObject* string);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetStringSize
// Desc: Returns the size in bytes of the specified string.
// Parm: string - A TNIString object, or nullptr.
// Retn: size_t - The size in bytes of the text string. Note that this is the
//       size of the UTF-8 encoded text data, not including zero termination,
//       and not including any overheads. If the string is null or empty, this
//       returns zero.
// ============================================================================
size_t TNIGetStringSize(const TNIString* string);



// ============================================================================
// Name: TNISetStringText
// Desc: Modifies the TNIString to a copy of the supplied UTF-8 encoded text
//       buffer.
// Parm: string - The existing TNIString object to modify. Modifying the string
//       invalidates any pointer previously returned by TNIGetStringText().
// Parm: utf8text - The UTF-8 text buffer. The contents of the buffer are copied
//       into the TNIString. The function results are undefined if this
//       parameter does not represent a valid UTF-8 encoded string.
// Note: If 'string' is a nullptr, this function does nothing.
// Note: If 'utf8text' is a nullptr, it is considered an empty string.
// ============================================================================
void TNISetStringText(TNIString* string, const char* utf8text);



// ============================================================================
// Name: TNIAppendStringText
// Desc: Appends the supplied UTF-8 encoded text buffer to the TNIString.
// Parm: string - The existing TNIString object to modify. Modifying the string
//       invalidates any pointer previously returned by TNIGetStringText().
// Parm: utf8text - The UTF-8 text buffer. The contents of the buffer are copied
//       into the TNIString. The function results are undefined if this
//       parameter does not represent a valid UTF-8 encoded string.
// Note: If 'string' is a nullptr, this function does nothing.
// Note: If 'utf8text' is a nullptr, this function does nothing.
// ============================================================================
void TNIAppendStringText(TNIString* string, const char* utf8text);



// ============================================================================
// TNIInteger functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIInteger* TNIAllocInteger(int64_t value);
int64_t TNIGetIntegerValue(const TNIObject* obj);
int64_t TNIGetIntegerValueRelease(const TNIObject* obj);
void TNISetIntegerValue(TNIInteger* obj, int64_t value);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNIFloat functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIFloat* TNIAllocFloat(double value);
double TNIGetFloatValue(const TNIObject* obj);
double TNIGetFloatValueRelease(const TNIObject* obj);
void TNISetFloatValue(TNIFloat* obj, double value);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNIAssetID functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIAssetID* TNIAllocAssetID(uint32_t userId, uint32_t contentId, uint32_t version);
uint32_t TNIGetAssetIDUserID(const TNIAssetID* assetID);
uint32_t TNIGetAssetIDContentID(const TNIAssetID* assetID);
uint32_t TNIGetAssetIDVersion(const TNIAssetID* assetID);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNICoordinates functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNICoordinates* TNIAllocCoordinatesWorldCoordinates(int16_t h, int16_t v, float x, float y, float z);
TNICoordinates* TNIAllocCoordinatesFloatingPoint(double x, double y, double z);
void TNICoordinatesSubtract(const TNICoordinates* lhs, const TNICoordinates* rhs, float* o_x, float* o_y, float* o_z);
bool TNICoordinatesEquals(const TNICoordinates* lhs, const TNICoordinates* rhs);
void TNICoordinatesGetWorldCoordinates(const TNICoordinates* coords, int16_t* o_h, int16_t* o_v, float* o_x, float* o_y, float* o_z);
void TNICoordinatesGetFloatingPoint(const TNICoordinates* coords, double* o_x, double* o_y, double* o_z);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNIArray functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIArray* TNIAllocArray(size_t arraySize);
void TNISetArraySize(TNIArray* array, size_t newArraySize);
size_t TNIGetArraySize(const TNIObject* array);
void TNISetArrayElement(TNIArray* array, size_t arrayIndex, const TNIObject* value);
TNIObject* TNIGetArrayElement(const TNIObject* array, size_t arrayIndex);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNISoup functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNISoup* TNIAllocSoup(void);
void TNISetSoupKeyValue(TNISoup* soup, const TNILabel* key, TNIObject* object);
const TNIObject* TNIGetSoupValueByKey(const TNISoup* soup, const TNILabel* key);
size_t TNIGetSoupKeyCount(const TNISoup* soup);
const TNILabel* TNIGetSoupKeyByIndex(const TNISoup* soup, size_t index);
const TNIObject* TNIGetSoupValueByIndex(const TNISoup* soup, size_t index);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// TNIVector functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIVector* TNIAllocVector(const float* elementData, size_t elementCount);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP
const float* TNIGetVectorData(const TNIVector* vectorObject);
size_t TNIGetVectorSize(const TNIVector* vectorObject);
float TNIGetVectorElement(const TNIVector* vectorObject, size_t elementIndex);


// ============================================================================
// TNIHandle functions.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIHandle* TNIAllocHandle(TNIHandleTarget* target);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP
bool TNIHandleEquals(const TNIHandle* lhs, const TNIHandle* rhs);


// ============================================================================
// HELPER FUNCTIONS
// ============================================================================
// The "cast" functions return the passed-in object cast to the appropriate
// type, or nullptr if the passed-in object is not of the appropriate type.
// The parameter object is effectively released.

// TODO: the semantics on these functions is wrong. We should remove them and
// replace them with renamed variants that behave as expected.

#if !TRAINZ_NATIVE_INTERFACE_CPP
const TNIString* TNICastString(const TNIObject* object);
const TNIInteger* TNICastInteger(const TNIObject* object);
const TNIFloat* TNICastFloat(const TNIObject* object);
const TNIAssetID* TNICastAssetID(const TNIObject* object);
const TNIArray* TNICastArray(const TNIObject* object);
const TNIVector* TNICastVector(const TNIObject* object);
const TNISoup* TNICastSoup(const TNIObject* object);
const TNILibrary* TNICastLibrary(const TNIObject* object);
const TNILabel* TNICastLabel(const TNIObject* object);
const TNICoordinates* TNICastCoordinates(const TNIObject* object);
const TNIHandle* TNICastHandle(const TNIObject* object);
const TNIGameObjectID* TNICastGameObjectID(const TNIObject* object);

const TNIString* TNICastStringRelease(const TNIObject* object);
TNIInteger* TNICastMutableIntegerRelease(TNIObject* object);
const TNIInteger* TNICastIntegerRelease(const TNIObject* object);
TNIFloat* TNICastMutableFloatRelease(TNIObject* object);
const TNIFloat* TNICastFloatRelease(const TNIObject* object);
const TNIAssetID* TNICastAssetIDRelease(const TNIObject* object);
TNIArray* TNICastMutableArrayRelease(TNIObject* object);
const TNIArray* TNICastArrayRelease(const TNIObject* object);
const TNIVector* TNICastVectorRelease(const TNIObject* object);
const TNISoup* TNICastSoupRelease(const TNIObject* object);
const TNILibrary* TNICastLibraryRelease(const TNIObject* object);
const TNILabel* TNICastLabelRelease(const TNIObject* object);
const TNICoordinates* TNICastCoordinatesRelease(const TNIObject* object);
const TNIHandle* TNICastHandleRelease(const TNIObject* object);
const TNIGameObjectID* TNICastGameObjectIDRelease(const TNIObject* object);


TNIArray* TNIAllocArrayWith1(const TNIObject* object0);
TNIArray* TNIAllocArrayWith2(const TNIObject* object0, const TNIObject* object1);
TNIArray* TNIAllocArrayWith3(const TNIObject* object0, const TNIObject* object1, const TNIObject* object2);
TNIArray* TNIAllocArrayWith4(const TNIObject* object0, const TNIObject* object1, const TNIObject* object2, const TNIObject* object3);
TNIArray* TNIAllocArrayWith5(const TNIObject* object0, const TNIObject* object1, const TNIObject* object2, const TNIObject* object3, const TNIObject* object4);
TNIArray* TNIAllocArrayWith6(const TNIObject* object0, const TNIObject* object1, const TNIObject* object2, const TNIObject* object3, const TNIObject* object4, const TNIObject* object5);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIAllocStringWithIntegerValue
// Desc: Allocates a TNIString which represents a specified integer value.
// Parm: number - An integer value which will be converted to string form.
// Retn: TNIString - The newly allocated TNIString.
// Note: You must eventually release the allocated string using TNIRelease().
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIString* TNIAllocStringWithIntegerValue(int number);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetStringIntegerValue
// Desc: Parses an integer value from the specified string.
// Parm: string - A string object for which you have a reference. You must not
//       call this function if you do not own a reference to the string.
// Parm: failureSignal - An integer value which is returned if parsing fails.
// Retn: int - The integer value parsed from the string, or 'failureSignal' if
//       parsing failed for any reason.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
int TNIGetStringIntegerValue(const TNIObject* string, int failureSignal = 0);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP


// ============================================================================
// Name: TNIGetStringIntegerValueAndRelease
// Desc: As TNIGetStringIntegerValue, but releases a single reference to the
//       passed string after parsing.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
int TNIGetStringIntegerValueAndRelease(const TNIObject* string, int failureSignal = 0);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// END COMPILER OPTIONS
// ============================================================================
#ifdef __cplusplus
}
#endif // __cplusplus

#endif // _TNIFunctions_H
