// ============================================================================
// File: TNIFunctions.hpp
// Desc: C++11 specific helper wrappers around the (language-agnostic) Trainz
//       Native Interface API. This functionality is provided to simplify 
//       programming TNI DLLs in the C++11 language.
// ============================================================================
#ifndef _TNIFunctions_HPP
#define _TNIFunctions_HPP
#pragma once

#include "trainznativeinterface.hpp"
#include "TNIFunctions.h"
#include <stdarg.h>


//=============================================================================
// CPP interface.
//=============================================================================
namespace TNI
{
  TNIRef<TNIString> TNIAllocString(const char* utf8text);
  TNIRef<TNIInteger> TNIAllocInteger(int64_t value);
  int64_t TNIGetIntegerValue(TNIRefParam<const TNIObject> obj);
  void TNISetIntegerValue(TNIRefParam<TNIInteger> obj, int64_t value);
  TNIRef<TNIFloat> TNIAllocFloat(double value);
  double TNIGetFloatValue(TNIRefParam<const TNIObject> obj);
  void TNISetFloatValue(TNIRefParam<TNIFloat> obj, double value);
  TNIRef<TNIAssetID> TNIAllocAssetID(uint32_t userID, uint32_t contentID, uint32_t version);
  uint32_t TNIGetAssetIDUserID(TNIRefParam<const TNIAssetID> assetID);
  uint32_t TNIGetAssetIDContentID(TNIRefParam<const TNIAssetID> assetID);
  uint32_t TNIGetAssetIDVersion(TNIRefParam<const TNIAssetID> assetID);
  TNIRef<TNICoordinates> TNIAllocCoordinatesWorldCoordinates(int16_t h, int16_t v, float x, float y, float z);
  TNIRef<TNICoordinates> TNIAllocCoordinatesFloatingPoint(double x, double y, double z);
  void TNICoordinatesSubtract(TNIRefParam<const TNICoordinates> lhs, TNIRefParam<const TNICoordinates> rhs, float* o_x, float* o_y, float* o_z);
  bool TNICoordinatesEquals(TNIRefParam<const TNICoordinates> lhs, TNIRefParam<const TNICoordinates> rhs);
  void TNICoordinatesGetWorldCoordinates(TNIRefParam<const TNICoordinates> coords, int16_t* o_h, int16_t* o_v, float* o_x, float* o_y, float* o_z);
  void TNICoordinatesGetFloatingPoint(TNIRefParam<const TNICoordinates> coords, double* o_x, double* o_y, double* o_z);
  TNIRef<TNIArray> TNIAllocArray(size_t arraySize);
  TNIRef<TNIObject> TNIGetArrayElement(TNIRefParam<const TNIObject> array, size_t arrayIndex);
  void TNISetArraySize(TNIRefParam<TNIArray> array, size_t newArraySize);
  size_t TNIGetArraySize(TNIRefParam<const TNIObject> array);
  void TNISetArrayElement(TNIRefParam<TNIArray> array, size_t arrayIndex, TNIRefParam<const TNIObject> value);
  TNIRef<TNISoup> TNIAllocSoup(void);
  void TNISetSoupKeyValue(TNIRefParam<TNISoup> soup, TNIRefParam<const TNILabel> key, TNIRefParam<TNIObject> object);
  size_t TNIGetSoupKeyCount(TNIRefParam<const TNISoup> soup);
  TNIRef<const TNIObject> TNIGetSoupValueByKey(TNIRefParam<const TNISoup> soup, TNIRefParam<const TNILabel> key);
  TNIRef<const TNILabel> TNIGetSoupKeyByIndex(TNIRefParam<const TNISoup> soup, size_t index);
  TNIRef<const TNIObject> TNIGetSoupValueByIndex(TNIRefParam<const TNISoup> soup, size_t index);
  TNIRef<TNIVector> TNIAllocVector(const float* elementData, size_t elementCount);
  TNIRef<TNIHandle> TNIAllocHandle(TNIRefParam<TNIHandleTarget> target);

  TNIRef<const TNIString> TNICastString(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIInteger> TNICastInteger(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIFloat> TNICastFloat(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIAssetID> TNICastAssetID(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIArray> TNICastArray(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIVector> TNICastVector(TNIRefParam<const TNIObject> object);
  TNIRef<const TNISoup> TNICastSoup(TNIRefParam<const TNIObject> object);
  TNIRef<const TNILibrary> TNICastLibrary(TNIRefParam<const TNIObject> object);
  TNIRef<const TNILabel> TNICastLabel(TNIRefParam<const TNIObject> object);
  TNIRef<const TNICoordinates> TNICastCoordinates(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIHandle> TNICastHandle(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIBuffer> TNICastBuffer(TNIRefParam<const TNIObject> object);
  TNIRef<const TNIGameObjectID> TNICastGameObjectID(TNIRefParam<const TNIObject> object);
  
  TNIRef<TNIInteger> TNICastMutableInteger(TNIRefParam<const TNIObject> object);
  TNIRef<TNIFloat> TNICastMutableFloat(TNIRefParam<const TNIObject> object);
  TNIRef<TNIArray> TNICastMutableArray(TNIRefParam<const TNIObject> object);
  TNIRef<TNIStream> TNICastMutableStream(TNIRefParam<const TNIObject> object);

  TNIRef<TNIArray> TNIAllocArrayWith1(TNIRefParam<const TNIObject> object0);
  TNIRef<TNIArray> TNIAllocArrayWith2(TNIRefParam<const TNIObject> object0, TNIRefParam<const TNIObject> object1);
  TNIRef<TNIArray> TNIAllocArrayWith3(TNIRefParam<const TNIObject> object0, TNIRefParam<const TNIObject> object1, TNIRefParam<const TNIObject> object2);
  TNIRef<TNIArray> TNIAllocArrayWith4(TNIRefParam<const TNIObject> object0, TNIRefParam<const TNIObject> object1, TNIRefParam<const TNIObject> object2, TNIRefParam<const TNIObject> object3);
  TNIRef<TNIArray> TNIAllocArrayWith5(TNIRefParam<const TNIObject> object0, TNIRefParam<const TNIObject> object1, TNIRefParam<const TNIObject> object2, TNIRefParam<const TNIObject> object3, TNIRefParam<const TNIObject> object4);
  TNIRef<TNIArray> TNIAllocArrayWith6(TNIRefParam<const TNIObject> object0, TNIRefParam<const TNIObject> object1, TNIRefParam<const TNIObject> object2, TNIRefParam<const TNIObject> object3, TNIRefParam<const TNIObject> object4, TNIRefParam<const TNIObject> object5);

  TNIRef<TNIString> TNIAllocStringWithIntegerValue(int number);
  int TNIGetStringIntegerValue(TNIRefParam<const TNIObject> string, int failureSignal = 0);
  
  const char* TNIGetStringText(TNIRefParam<const TNIObject> string);
  
  uint32_t TNIGetObjectClass(TNIRefParam<const TNIObject> object);
  
  void TNILog(TNIRefParam<const TNIContext> context, const char* textStringUTF8);
  void TNILogObject(TNIRefParam<const TNIContext> context, TNIRefParam<const TNIObject> object, size_t indentLevel = 0);
  void TNILogf(TNIRefParam<const TNIContext> context, const char* formatStringUTF8, ...);
  void TNILogv(TNIRefParam<const TNIContext> context, const char* formatStringUTF8, va_list argv);
  
  TNIRef<TNIBuffer> TNIAllocBuffer(const char* copyFromData, size_t dataSize);
  char* TNIBufferGetPointer(TNIRefParam<const TNIBuffer> buffer);
  size_t TNIBufferGetByteLength(TNIRefParam<const TNIBuffer> buffer);
  
  uint32_t TNIGetLibraryVersion(TNIRefParam<const TNILibrary> library);
};
using namespace TNI;




#endif // _TNIFunctions_HPP
