// ============================================================================
// File: TNILibrary.h
// Desc: A TNILibrary is roughly equivalent to a trainzscript "Library" class. 
//       A third-party DLL should return a new TNILibrary instance from the  
//       initial entry-point call, to open a communication channel between 
//       Trainz and the third-party DLL.
// ============================================================================
#ifndef _TNILIBRARY_H
#define _TNILIBRARY_H
#pragma once

#include "trainznativeinterface.h"
class TNILabel;

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus



// ============================================================================
// Name: TNILibraryInitData
// Desc: An initialisation structure used when creating a TNILibrary. This
//       structure is not a TNIObject and may be safely constructed by the 
//       third-party developer.
// ============================================================================
class TNILibraryInitData
{
protected:

  uint32_t      m_structVersion;            // The struct format version, set during construction.

public:

  // The LibraryCall function pointer type.
  typedef TNIObject* (*LibraryCall)(TNIContext* context, const TNILibrary* library, const TNILabel* functionName, const TNIObject* params);

  // The ShutdownCall function pointer type.
  typedef void (*ShutdownCall)(TNIContext* context, TNILibrary* library);

  uint32_t      m_libraryVersion;       // The version of the library, as determined by the library developer. 
  LibraryCall   m_libraryCall;          // A pointer to the function which services library calls for this library.
  ShutdownCall  m_shutdownCall;         // A pointer to the function which is called when the library is shut down.


  // The default constructor; this will initialise all fields appropriately, such that
  // the developer need only specify fields which they wish to override.
  inline TNILibraryInitData(void)
    : m_structVersion(0)
    , m_libraryVersion(0)
    , m_libraryCall(NULL)
    , m_shutdownCall(NULL)
  {
  }

};



// ============================================================================
// Name: TNILibrary
// Desc: Roughly analogous to the trainzscript "Library" class, this object
//       provides a mechanism for code to make function calls against the
//       function pointers supplied to TNIAllocLibrary().
// ============================================================================
class TNILibrary : public TNIObject
{
protected:

};


// ============================================================================
// ACCESSOR FUNCTIONS
// ============================================================================



// ============================================================================
// Name: TNIAllocLibrary
// Desc: Allocate a new library in the specified context, given a unique 
//       library name and the initialisation data.
// Parm: context - The context in which this library is being created.
// Parm: libraryName - The name of this library, which frequently matches the
//       name of the plugin in which it is implemented (although it is possible
//       for a single plugin to export multiple libraries). Each library
//       created in a given context must have a unique name.
// Parm: libraryInitData - A TNILibraryInitData structure which describes
//       the library to be created.
// Retn: TNILibrary* - A valid object on success, or NULL on failure.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNILibrary* TNIAllocLibrary(TNIContext* context, const TNILabel* libraryName, const TNILibraryInitData& libraryInitData);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetLibraryVersion
// Desc: Returns the developer-specified version for the given library.
// Parm: library - The library to query.
// Retn: uint32_t - the developer-specified version for the given library. 
//       Returns zero on failure.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
uint32_t TNIGetLibraryVersion(const TNILibrary* library);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetLibraryAsset
// Desc: Returns the AssetID for the asset this library belongs to. Note that
//       Builtin libraries such as TNITrainzScript or TNISocket may not have an
//       owning asset, and will thus return null.
// Parm: library - The library to query.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
const TNIAssetID* TNIGetLibraryAsset(const TNILibrary* library);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNIGetLibraryConfigSoup
// Desc: Returns a config soupd for this library belongs to, if any.
// Parm: library - The library to query.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
const TNISoup* TNIGetLibraryConfigSoup(const TNILibrary* library);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP



// ============================================================================
// Name: TNICallLibraryFunction
// Desc: Calls into the specified library's LibraryCall function.
// Parm: library - The library on which to make the call.
// Parm: functionName - The name of the function to call. Must be a non-empty 
//       string.
// Parm: params - NULL, or a TNIObject object.
// Retn: TNIObject* - The return value from the LibraryCall function, which may
//       be NULL or a valid TNIObject. Returns NULL on failure. It is the 
//       caller's responsibility to release the returned object.
// ============================================================================
#if !TRAINZ_NATIVE_INTERFACE_CPP
TNIObject* TNICallLibraryFunction(const TNILibrary* library, const TNILabel* functionName, const TNIObject* params);
#endif // !TRAINZ_NATIVE_INTERFACE_CPP




// ============================================================================
// END COMPILER OPTIONS
// ============================================================================
#ifdef __cplusplus
}
#endif // __cplusplus

#endif // _TNILIBRARY_H
