//=============================================================================
// File: TNIPhysics.h
// Desc: Declares the TNI physics associated classes, enums and functions.
//=============================================================================
#ifndef _TNIPhysics_H_
#define _TNIPhysics_H_

#include "trainznativeinterface.h"

class TNILabel;


//=============================================================================
// COMPILER OPTIONS
//
// Despite being written in C++, this interface is considered portable. Third-
// party developers are thus able to access the trainznativeinterface.dll 
// from any native language which is capable of building a DLL, by building an
// alternative header in that language.
// 
// Exceptions must not be thrown in TNI DLL code.
//=============================================================================
#ifdef __cplusplus
extern "C" {
#endif // __cplusplus


class TNIStream;
class TNIPhysicsVehicle;
class TNIPhysicsVehicleFrameSnapshot;
class TNIPhysicsEngineSpec;
class TNIGameObjectID;



//=============================================================================
// Name: TNIPhysicsPluginCommand
// Desc: Enumeration listing the possible physics commands the game may issue
//       to a plugin. Commands and parameters will be written, in sequence, to
//       the TNIStream param passed to the ProcessPhysicsCommands library call.
//
//       Physics plugins must implement code to read all of these commands and
//       their parameters, even if the plugin does not handle that specific
//       command. Failure to correctly parse the command stream will result in
//       data corruption, and undefined plugin behaviour.
//
//=============================================================================
enum TNIPhysicsPluginCommand
{
  TNIP_PluginCommandNone                    = 0,

  //=============================================================================
  // Desc: The main physics update function. This is regularly called by Trainz
  //       code to update the traincar physics for a particular time period.
  //       This function takes two extra params for the front and rear coupled
  //       vehicles, but keep in mind these may be using different TNI plugins.
  // Parm: float - The time interval to update for.
  // Parm: TNIPhysicsVehicle - The front coupled vehicle.
  // Parm: TNIPhysicsVehicle - The rear coupled vehicle.
  //=============================================================================
  TNIP_UpdatePhysicsForVehicle              = 1,

  //=============================================================================
  // Desc: Called from Trainz to notify of an adjustment to a vehicles momentum.
  //       This is typically used to apply the results of coupler physics
  //       interactions. Vehicles for which the plugin does not internally track
  //       momentum can safely ignore this call.
  // Parm: double - The new vehicle momentum.
  //=============================================================================
  TNIP_AdjustMomentum                       = 2,

  //=============================================================================
  // Desc: Called by Trainz to notify that all updates relating to this vehicle
  //       for the current frame are complete. This includes all plugin updates
  //       and all coupler interactions. Plugins may use this as an appropriate
  //       location to synchronise plugin state to that of the game, or apply
  //       visible state changes only. At the time this call is made it is "too
  //       late" to perform or apply physics calculations for a given frame.
  // Parm: float - The time interval that's been updated.
  // Parm: double - The final momentum for this frame.
  //=============================================================================
  TNIP_FinaliseFramePhysics                 = 3,

  //=============================================================================
  // Desc: Called following an external physics state update in order to perform
  //       any physics-mode-specific follow up validation/correction. Examples
  //       include after physics/control state replication in multiplayer.
  //=============================================================================
  TNIP_NormalisePhysicsData                 = 4,


  //=============================================================================
  // Desc: Called by Trainz code to request save data for a specific vehicle.
  //       If the plugin maintains any internal state for the specified vehicle
  //       then this function should save that state, so that players are able to
  //       save and reload with seemless physics behaviour. The save state should
  //       be returned to the game by adding a TNIP_ReturnSaveState command to
  //       the library call result stream (see TNIPhysicsGameCommand for more).
  //       If the plugin tracks no state for the vehicle passed, then this call
  //       can be safely ignored.
  //=============================================================================
  TNIP_SaveVehicleState                     = 101,

  //=============================================================================
  // Desc: Called by Trainz code to load previously saved vehicle state. Note
  //       that this is always called, even if the plugin does not produce any
  //       save state when TNIP_SaveVehicleState is called. Also note that the
  //       the passed soup param may have been produced by a different plugin.
  // Parm: TNISoup - A saved soup generated by a previous TNIP_SaveVehicleState
  //       call on this or another plugin. The soup will contain an AssetID param
  //       for the plugin that produced it, using the tag name "plugin-kuid".
  //=============================================================================
  TNIP_LoadVehicleState                     = 102,


  //=============================================================================
  // Desc: Called by Trainz code to add coal to a steam engine. For example, when
  //       the player uses the manual "Shovel coal" command.
  // Parm: double - The volume (not mass) of coal to add.
  //=============================================================================
  TNIP_SteamEngine_AddCoalToFire            = 201,


  //=============================================================================
  // Desc: Called from script to execute custom behaviour within the plugin. This
  //       will be added when a script calls Vehicle.LibraryCallPhysicsPlugin().
  // Parm: TNILabel - The name of the plugin function to execute. If the function
  //       name is unrecognised or unsupported, the plugin should log an error
  //       and ignore it. (The TNISoup param must still be read though, to avoid
  //       the command stream being corrupted.)
  // Parm: TNISoup - An optional TNISoup to be included with the command. This is
  //       used to pass more parameters to the custom plugin function, where
  //       required. Parsing code must handle the possibility that the parameter
  //       is null, even if the extra parameters are not optional for the plugin
  //       code. Similarly, plugins must not make assumptions about the validity
  //       of the soup data, and should either appropriately handle all inputs or
  //       ignore the command entirely.
  //=============================================================================
  TNIP_ScriptFunctionCall                   = 2001,

};


//=============================================================================
// Name: TNIPhysicsGameCommand
// Desc: Enumeration listing the possible physics commands a plugin may issue
//       to the game. A TNIPhysics plugin may write these commands (and their
//       params) to the TNIStream returned by a ProcessPhysicsCommands library
//       call.
//=============================================================================
enum TNIPhysicsGameCommand
{
  TNIP_GameCommandNone                        = 0,

  //=============================================================================
  // Desc: Specifies which vehicle a sequence of update commands refers to. Use
  //       of this command is not necessary, and by default any written commands
  //       will update the vehicle which ProcessPhysicsCommands() was called for.
  //       If, however, the plugin wishes to run commands for connection data on
  //       one of the coupled vehicles (such as brake pipes, etc) then this is
  //       used to specify that vehicle.
  // Parm: TNIPhysicsVehicle - The vehicle to start updating.
  //=============================================================================
  TNIP_UpdateVehicleBegin                     = 1,

  //=============================================================================
  // Desc: Used to mark the end of a vehicle update stream. This will set the
  //       update vehicle to null. Following this, a new TNIP_UpdateVehicleBegin
  //       command *must* be written before any further update commands.
  //=============================================================================
  TNIP_UpdateVehicleEnd                       = 2,

  //=============================================================================
  // Desc: Used to return the generated save state for a TNIP_SaveVehicleState
  //       command (documented above, in TNIPhysicsPluginCommand).
  // Parm: TNISoup - A soup containing the saved plugin state. The plugin should
  //       write enough state to adequately restore it's current internal state
  //       for the given vehicle on reload.
  //=============================================================================
  TNIP_ReturnSaveState                        = 3,


  TNIP_EngineCommandBegin                     = 100,  // Marker only, not a valid command.


  //=============================================================================
  // Desc: Sets the current resistant force for train movement. This includes all
  //       track resistance, slope resistance, and internal engine resistances.
  // Parm: double - The current resistance force, in Newtons.
  //=============================================================================
  TNIP_SetResistanceForce                     = TNIP_EngineCommandBegin + 1,

  //=============================================================================
  // Desc: Sets the current braking force for the train. This is the force from
  //       any/all applied brakes.
  // Parm: double - The current braking force, in Newtons.
  //=============================================================================
  TNIP_SetBrakingForce                        = TNIP_EngineCommandBegin + 2,

  //=============================================================================
  // Desc: Sets the force currently being output by the engine.
  // Parm: double - The current engine force, in Newtons.
  //=============================================================================
  TNIP_SetEngineForce                         = TNIP_EngineCommandBegin + 3,

  //=============================================================================
  // Desc: Sets the force currently being applied to the wheels by the engine.
  //       In other words, this is the engine force minus any wheelslip.
  // Parm: double - The current applied engine force, in Newtons.
  //=============================================================================
  TNIP_SetAppliedEngineForce                  = TNIP_EngineCommandBegin + 4,

  //=============================================================================
  // Desc: Sets the amount of force currently being lost to wheelslip.
  // Parm: double - The current wheelslip force, in Newtons.
  //=============================================================================
  TNIP_SetWheelslipForce                      = TNIP_EngineCommandBegin + 5,

  //=============================================================================
  // Desc: Updates the current drawn stat.
  // Parm: double - The current being drawn, in Amps.
  //=============================================================================
  TNIP_SetCurrentDrawn                        = TNIP_EngineCommandBegin + 6,

  //=============================================================================
  // Desc: Updates the "energy use" stat, which is used to generate the PFX
  //       "load" variable. This has no effect on traincar physics, just PFX.
  // Parm: double - The energy use for the current time interval.
  //=============================================================================
  TNIP_SetPFXEnergyUseAmount                  = TNIP_EngineCommandBegin + 7,

  //=============================================================================
  // Desc: Sets the current forward momentum for a traincar.
  // Parm: double - The new momentum for the updated traincar, in Newton seconds.
  //=============================================================================
  TNIP_SetWheelMomentum                       = TNIP_EngineCommandBegin + 8,

  //=============================================================================
  // Desc: Adds momentum to the updated traincar. A negative (or positive, for
  //       backward travelling traincars) value may be used to remove momentum.
  // Parm: double - The momentum to add onto the traincar, in Newton seconds.
  //=============================================================================
  TNIP_AddWheelMomentum                       = TNIP_EngineCommandBegin + 9,

  //=============================================================================
  // Desc: Sets the amount of 'extra' engine mass a vehicle currently has. This
  //       is typcially the fuel mass (water and coal) for steam locos.
  // Parm: double - The current extra engine mass, in kg.
  //=============================================================================
  TNIP_SetExtraEngineMass                     = TNIP_EngineCommandBegin + 10,

  //=============================================================================
  // Desc: Sets the value of a named engine param, as returned by the script
  //       GetEngineParam() function on Vehicle. This is used to add supports for
  //       custom named parameters, attempting to override the natively supported
  //       params may not work.
  // Parm: TNILabel - The name of the engine parameter to add/update.
  // Parm: double - The new value of the parameter.
  //=============================================================================
  TNIP_SetEngineParam                         = TNIP_EngineCommandBegin + 11,

  //=============================================================================
  // Desc: Sets the current engine throttle. This should gradually move toward
  //       the throttle setting, with some amount of latency.
  // Parm: double - The new engine throttle amount (0 to 1).
  //=============================================================================
  TNIP_SetEngineThrottle                      = TNIP_EngineCommandBegin + 12,

  //=============================================================================
  // Desc: Sets whether the brake system compressor is active.
  // Parm: int8_t - The new compressor state (0- off, 1-on).
  //=============================================================================
  TNIP_SetCompressorActive                    = TNIP_EngineCommandBegin + 13,

  //=============================================================================
  // Desc: Sets the pressure of the brake system equaliser.
  // Parm: double - The new equaliser pressure, in g/m^3.
  //=============================================================================
  TNIP_SetEqualiserPressure                   = TNIP_EngineCommandBegin + 14,

  //=============================================================================
  // Desc: Sets the pressure of the brake system main reservoir.
  // Parm: double - The new main reservoir pressure, in g/m^3.
  //=============================================================================
  TNIP_SetMainReservoirPressure               = TNIP_EngineCommandBegin + 15,

  //=============================================================================
  // Desc: Sets the pressure of the brake system auxiliary reservoir.
  // Parm: double - The new auxiliary reservoir pressure, in g/m^3.
  //=============================================================================
  TNIP_SetAuxReservoirPressure                = TNIP_EngineCommandBegin + 16,

  //=============================================================================
  // Desc: Sets the pressure of the brake cylinder.
  // Parm: double - The new brake cylinder pressure, in g/m^3.
  //=============================================================================
  TNIP_SetAutoBrakeCylinderPressure           = TNIP_EngineCommandBegin + 17,

  //=============================================================================
  // Desc: Sets the pressure of the independent/loco brake cylinder.
  // Parm: double - The new independent/loco brake cylinder pressure, in g/m^3.
  //=============================================================================
  TNIP_SetIndependentBrakeCylinderPressure    = TNIP_EngineCommandBegin + 18,

  //=============================================================================
  // Desc: Sets the pressure of the main train brake pipe.
  // Parm: double - The new train brake pipe pressure, in g/m^3.
  //=============================================================================
  TNIP_SetTrainBrakePipePressure              = TNIP_EngineCommandBegin + 19,

  //=============================================================================
  // Desc: Sets the pressure of the electro-pneumatic brake pipe.
  // Parm: double - The new electro-pneumatic brake pipe pressure, in g/m^3.
  //=============================================================================
  TNIP_SetEpReservoirPipePressure             = TNIP_EngineCommandBegin + 20,

  //=============================================================================
  // Desc: Sets the pressure of the no3 brake pipe.
  // Parm: double - The new no3 brake pipe pressure, in g/m^3.
  //=============================================================================
  TNIP_SetNo3PipePressure                     = TNIP_EngineCommandBegin + 21,

  //=============================================================================
  // Desc: Sets the pressure of the no4 brake pipe.
  // Parm: double - The new no4 brake pipe pressure, in g/m^3.
  //=============================================================================
  TNIP_SetNo4PipePressure                     = TNIP_EngineCommandBegin + 22,

  //=============================================================================
  // Desc: Notifies of the brake pressure flow rate.
  // Parm: double - The new brake pressure flow rate, in g/m^3.
  //=============================================================================
  TNIP_SetBrakePressureFlowRate               = TNIP_EngineCommandBegin + 23,


  TNIP_EngineCommandEnd                       = 499,  // Marker only, not a valid command.
  TNIP_SteamEngineCommandBegin                = 500,  // Marker only, not a valid command.


  //=============================================================================
  // Desc: Sets the coal level in a steam locomotive. This is the current ratio
  //       between the minimum (0) and maximum (1) coal levels for the firebox.
  //       ie, a reading of 0 indicates the fire is at minimum coal, not that
  //       it's necessarily "empty", and a value over 1 indicates the fire is
  //       over-filled with coal.
  // Parm: double - The current coal level in the firebox.
  //=============================================================================
  TNIP_SetCoalLevel                           = TNIP_SteamEngineCommandBegin + 1,

  //=============================================================================
  // Desc: Sets the water level in a steam locomotive. This is the current ratio
  //       between the minimum (0) and maximum (1) water levels for the boiler,
  //       as would be shown on the cab gauge(s). ie, a reading of 0 indicates
  //       the boiler is at or below minimum, not that it's necessarily "empty".
  // Parm: double - The current water level in the boiler.
  //=============================================================================
  TNIP_SetWaterLevel                          = TNIP_SteamEngineCommandBegin + 2,

  //=============================================================================
  // Desc: Sets the current pressure within the boiler, in pascals.
  // Parm: double - The new boiler pressure, in pascals.
  //=============================================================================
  TNIP_SetBoilerPressure                      = TNIP_SteamEngineCommandBegin + 3,

  //=============================================================================
  // Desc: Sets the current pressure within the steam chest, in pascals.
  // Parm: double - The new steam chest pressure, in pascals.
  //=============================================================================
  TNIP_SetSteamChestPressure                  = TNIP_SteamEngineCommandBegin + 4,

  //=============================================================================
  // Desc: Sets the amount the engine is being 'thrashed', as returned to script
  //       by a call to Locomotive.GetEngineParam("thrashing");
  // Parm: double - The amount the engine is being 'thrashed', 0 to 1 for normal
  //       engine operation, >1 for thrashing.
  //=============================================================================
  TNIP_SetThrashing                           = TNIP_SteamEngineCommandBegin + 5,

  //=============================================================================
  // Desc: Informs Trainz to consume an amount of coal product, for use as steam
  //       engine fuel. The caller should first verify that there is sufficient
  //       fuel to meet the request, using TNIPhysicsGetVehicleCoalAvailable().
  // Parm: int32_t - The amount of product to consume.
  //=============================================================================
  TNIP_ConsumeCoal                            = TNIP_SteamEngineCommandBegin + 6,

  //=============================================================================
  // Desc: Informs Trainz to consume an amount of water product, for use as steam
  //       engine fuel. The caller should first verify that there is sufficient
  //       fuel to meet the request, using TNIPhysicsGetVehicleWaterAvailable().
  // Parm: int32_t - The amount of product to consume.
  //=============================================================================
  TNIP_ConsumeWater                           = TNIP_SteamEngineCommandBegin + 7,


  TNIP_SteamEngineCommandEnd                  = 999,  // Marker only, not a valid command.
  TNIP_EffectCommandBegin                     = 1000, // Marker only, not a valid command.


  //=============================================================================
  // Desc: Sets the angular velocity of the wheels on a steam locomotive. This
  //       affects the value of the "cyclespeed2" PFX variable.
  // Parm: float - The angular velocity of the driving wheels, in m/s.
  //=============================================================================
  TNIP_SetWheelAngularVelocity                = TNIP_EffectCommandBegin + 1,

  //=============================================================================
  // Desc: Sets the animation data for the wheels of a traincar. Two values are
  //       always written here, but which is used will depend on the type of
  //       wheel animation which being used by the traincar.
  // Parm: float - The current wheel rotation, from 0 to 2PI.
  // Parm: float - The wheel velocity, in m/s.
  //=============================================================================
  TNIP_SetWheelAnimationData                  = TNIP_EffectCommandBegin + 2,

  //=============================================================================
  // Desc: Sets the volume of the brake release sound on this traincar.
  // Parm: float - The new brake release sound volume, 0 to 1.
  //=============================================================================
  TNIP_SetBrakeReleaseSoundFade               = TNIP_EffectCommandBegin + 3,

  //=============================================================================
  // Desc: Sets the value of a named PFX variable. See validation/steam-mode.txt
  //       for a list of valid variable names.
  // Parm: TNILabel - The name of the variable to update.
  // Parm: float - The new value of the variable.
  //=============================================================================
  TNIP_SetPFXVariableValue                    = TNIP_EffectCommandBegin + 4,

  TNIP_EffectCommandEnd                       = 1499, // Marker only, not a valid command.
  TNIP_ScriptCommandBegin                     = 2000, // Marker only, not a valid command.

  //=============================================================================
  // Desc: Specifies the custom features which this plugin supports, so that they
  //       can be queried by script. This API is optional, but plugins should
  //       normally specify at least one feature ID, to allow scripts to query
  //       which plugin is running.
  // Parm: TNIArray - An array of string IDs (using TNILabel) for the supported
  //       features. This may be empty or null, but it must be written to avoid
  //       corrupting the command stream. Each string ID should be prefixed with
  //       the plugins base (i.e. versionless) KUID, followed by an English human
  //       readable feature string.
  //       For example:  "<kuid:1111:2222>/diesel-consumption"
  // Note: When called, the supplied feature list will completely replace any
  //       existing feature list. I.e. You must always provide the complete list
  //       of supported features for the current vehicle.
  //=============================================================================
  TNIP_SetPluginFeatureSupport                = 2001,

  //=============================================================================
  // Desc: Post a message to the vehicle script. This can be used to trigger the
  //       execution of custom script code from the plugin, if any script
  //       supports it. The posted message will use the 'major' value of
  //       "TNIPhysicsPlugin". The 'minor' value is specified by the first
  //       parameter to this command.
  // Parm: TNIString - The value to use for Message.minor. Must not be empty.
  // Parm: TNISoup - An optional TNISoup to be included with the message. This
  //       may be null but it must be written to avoid command stream corruption.
  //=============================================================================
  TNIP_ScriptPostMessage                      = 2002,

  TNIP_ScriptCommandEnd                       = 2099, // Marker only, not a valid command.

};



//=============================================================================
// Desc: Example return results for the QueryProcessingPriority library call.
enum TNIPhysics_VehiclePriority : int8_t
{
  TNIPhysics_PriorityInvalid        = -127, // Never use this plugin.

  TNIPhysics_PriorityLowest         = -20,  // Only use this plugin if there's no other option.
  TNIPhysics_PriorityLow            = -10,  // Prefer not using this plugin, but it might work ok.
  TNIPhysics_PriorityDefault        = 0,    // This plugin can handle the request adequetly.
  TNIPhysics_PriorityHigh           = 10,   // This plugin is somewhat specialised to handle the request.
  TNIPhysics_PriorityHighest        = 20,   // This plugin was built exactly for this request.
};


//=============================================================================
enum TNIPhysics_TrainPhysicsModelSetting : int8_t
{
  TNIPhysics_TrainPhysicsNone       = 0,  // No control mode
  TNIPhysics_TrainPhysicsDCC        = 1,  // DCC/simple/easy control mode
  TNIPhysics_TrainPhysicsRealistic  = 2,  // Realistic/cab controls mode
};

//=============================================================================
enum TNIPhysics_VehicleEngineType : int8_t
{
  TNIPhysics_VehicleEngineNone      = 0,  // No engine, ie rolling stock
  TNIPhysics_VehicleEngineDiesel    = 1,  // Diesel/electric engine modelling
  TNIPhysics_VehicleEngineSteam     = 2,  // Steam engine modelling
};

//=============================================================================
enum TNIPhysics_TractionSetting
{
  TNIPhysics_TractionForward        = 1,  // Forward
  TNIPhysics_TractionNeutral        = 0,  // Neutral, engine is not engaged
  TNIPhysics_TractionReverse        = -1, // Reverse
};

//=============================================================================
enum TNIPhysics_TrainBrakeSetting
{
  TNIPhysics_TrainBrakeRelease      = 0,  // compressor -> brake
  TNIPhysics_TrainBrakeInitial      = 1, 
    // .. range ..
  TNIPhysics_TrainBrakeApplication  = 2,  // brake -> atmosphere
  TNIPhysics_TrainBrakeHandleOff    = 3,  // handle off (Application)
  TNIPhysics_TrainBrakeEmergency    = 4,  // brake -fast-> atmosphere

  TNIPhysics_TrainBrakeLap          = 5,
};

//=============================================================================
enum TNIPhysics_DynamicBrakeSetting
{
  TNIPhysics_DynamicBrakeTraction   = 0,  // Traction mode, power will go to the wheels
  TNIPhysics_DynamicBrakeNeutral    = 1,  // Neutral, engine is not engaged
  TNIPhysics_DynamicBrakeBrake      = 2,  // Braking mode, power will go to braking
};

//=============================================================================
enum TNIPhysics_EPBrakeSetting
{
  TNIPhysics_EPBrakeDisabled  = -1,
  TNIPhysics_EPBrakeApplied   = 0,
  TNIPhysics_EPBrakeNeutral   = 1,
  TNIPhysics_EPBrakeReleased  = 2,
};



//=============================================================================
// Name: TNIPhysicsVehicle
// Desc: A TNIPhysics representation of a single traincar/vehicle. Trainz will
//       allocate one TNIPhysicsVehicle object per traincar, which can then be
//       registered/unregistered from particular plugins at various times.
//       A TNI plugin programmer can safely store a reference to this vehicle
//       only if that vehicle is registered with the plugin. The reference must
//       be released when unregistered, or the programmer risks an access
//       violation.
//       The TNIPhysicsVehicle itself holds references to the AssetID used to
//       create the vehicle, the config database for that asset, an engine spec
//       structure describing the engine asset, and a few simple variables.
//       This data is accessible through various accessors detailed below, and
//       is safe to access at any time.
//       In addition, the vehicle will hold a reference to a frame snapshot
//       object within certain update calls. This frame data is cleared outside
//       of those specific calls, and attempts to access it at an inappropriate
//       time will result in access violations (ie, crashes). The exact times
//       when this data is present is documented on the relevant commands. If
//       a plugin is not running a command which explicitly states that the
//       frame data is present, then it must not attempt to access it.
//=============================================================================
class TNIPhysicsVehicle : public TNIObject
{
};



//=============================================================================
// Name: TNIPhysicsVehicleFrameSnapshot
// Desc: Holds data describing the physics state of a particular vehicle, for a
//       particular update call. An instance of this object is stored on the
//       vehicle during specific update calls. Attempts to access it outside
//       of such calls is unsafe and will likely result in access violations.
// Note: This object and the data it holds is not directly accessible. To read
//       the values here a plugin must use accessors detailed further down.
//=============================================================================
class TNIPhysicsVehicleFrameSnapshot : public TNIObject
{
};



//=============================================================================
// Name: TNIPhysicsEngineSpec
// Desc: Holds data describing the engine asset for a specific traincar. The
//       values on this class are cached within Route/Session assets to avoid
//       changes to an underlying engine from breaking Session behaviour, but
//       are otherwise pulled from the asset config. Consult the wiki for more
//       information on specific tags.
//=============================================================================
class TNIPhysicsEngineSpec : public TNIObject
{
};



//=============================================================================
// Name: TNIPhysicsEngineSpecSteam
// Desc: Extra parameters stored on TNIPhysicsEngineSpec for steam engines. An
//       attempt to access steam specific data on a non-steam engine will
//       result in access violation.
//       The values on this class are cached within Route/Session assets to
//       avoid changes to an underlying engine from breaking Session behaviour,
//       but are otherwise pulled from the asset config. Consult the wiki for
//       more information on specific tags.
//=============================================================================
class TNIPhysicsEngineSpecSteam : public TNIObject
{
};



//=============================================================================
const TNIPhysicsVehicle* TNICastPhysicsVehicle(const TNIObject* object);


//=============================================================================
// Name: TNIPhysicsGetVehicleID
// Desc: Returns the TNIGameObjectID for the vehicle passed. This can be used
//       to map script requests to TNI vehicle instances in custom functions.
// Parm: vehicle - The vehicle to query the ID for.
// Retn: TNIGameObjectID* - The ID for the vehicle passed. A reference will not
//       be added, so if you wish to store the object you must add a reference
//       yourself (or use TNIRef).
//=============================================================================
const TNIGameObjectID* TNIPhysicsGetVehicleID(const TNIPhysicsVehicle* vehicle);


//=============================================================================
// Name: TNIPhysicsGetTrainID
// Desc: Returns a TNIGameObjectID for the parent train/consist of the vehicle
//       passed. This can be used to check whether two vehicle instances are in
//       the same train, and/or to map train data provided from custom script
//       function calls. Keep in mind that vehicles may couple/decouple at any
//       time during gameplay, so it's not appropriate to cache this data for
//       long time periods.
// Parm: vehicle - The vehicle for which to query the ID of the parent train.
// Retn: TNIGameObjectID* - The ID for the vehicles train. A reference will not
//       be added, so if you wish to store the object you must add a reference
//       yourself (or use TNIRef).
//=============================================================================
const TNIGameObjectID* TNIPhysicsGetVehicleTrainID(const TNIPhysicsVehicle* vehicle);


//=============================================================================
// Name: TNIPhysicsDoesVehicleMatchID
// Desc: Returns whether a TNIPhysicsVehicle matches a TNIGameObjectID.
// Parm: vehicle - The vehicle to query for a match.
// Parm: id - The GameObjectID to query for a match.
//=============================================================================
bool TNIPhysicsDoesVehicleMatchID(const TNIPhysicsVehicle* vehicle, const TNIGameObjectID* id);


//=============================================================================
// Desc: Basic accessors for TNIPhysicsVehicle member variables, including
//       those on the frame data, engine spec or steam engine spec.
//=============================================================================


// AssetID/config accessors
const TNIAssetID* TNIPhysicsGetVehicleAssetID(const TNIPhysicsVehicle* vehicle);
const TNISoup* TNIPhysicsGetVehicleAssetConfig(const TNIPhysicsVehicle* vehicle);
const TNIAssetID* TNIPhysicsGetVehicleEngineAssetID(const TNIPhysicsVehicle* vehicle);
const TNISoup* TNIPhysicsGetVehicleEngineAssetConfig(const TNIPhysicsVehicle* vehicle);
const TNIAssetID* TNIPhysicsGetVehicleTrackAssetID(const TNIPhysicsVehicle* vehicle);
const TNISoup* TNIPhysicsGetVehicleTrackAssetConfig(const TNIPhysicsVehicle* vehicle);


// Train state accessors
double TNIPhysicsGetTrainLocoCount(const TNIPhysicsVehicle* vehicle);


// Train control settings
int8_t TNIPhysicsGetVehiclePhysicsModelSetting(const TNIPhysicsVehicle* vehicle);
int8_t TNIPhysicsGetVehicleTractionSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleThrottleSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleSteamRegulatorSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleSteamCutoffSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleTrainBrakeSetting(const TNIPhysicsVehicle* vehicle);
int8_t TNIPhysicsGetVehicleDynamicBrakeSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleLocoBrakeSetting(const TNIPhysicsVehicle* vehicle);
int8_t TNIPhysicsGetVehicleEpBrakeSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleBailTimeSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleHandBrakeSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleSandingSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleCoalRateSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleInjector1Setting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleInjector2Setting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleBlowerSetting(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleCustomSetting(const TNIPhysicsVehicle* vehicle, const TNILabel* settingName);


// Vehicle state accessors
double TNIPhysicsGetVehicleMass(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleVelocity(const TNIPhysicsVehicle* vehicle);
bool TNIPhysicsGetVehicleIsStopped(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleWheelMomentum(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleWheelslip(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleWheelslipTractionMultiplier(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleWheelslipMomentumMultiplier(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleEngineForce(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleEngineThrottle(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleThrottleGraphOutput(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleDynamicBrakeGraphOutput(const TNIPhysicsVehicle* vehicle);

bool TNIPhysicsGetVehicleIsDerailed(const TNIPhysicsVehicle* vehicle);
bool TNIPhysicsGetVehicleAllowsUserControl(const TNIPhysicsVehicle* vehicle);
bool TNIPhysicsGetVehicleIsAutomaticFireman(const TNIPhysicsVehicle* vehicle);
uint32_t TNIPhysicsGetVehicleCoalAvailable(const TNIPhysicsVehicle* vehicle);
uint32_t TNIPhysicsGetVehicleWaterAvailable(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleSandingTractionMultiplier(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleScriptTractionMultiplier(const TNIPhysicsVehicle* vehicle);


// Vehicle brake pressure state accessors
double TNIPhysicsGetVehicleEqualiserPressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleMainReservoirPressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleNo3PipePressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleNo4PipePressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleTrainBrakePipePressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleEpReservoirPipePressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleAuxReservoirPressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleAutoBrakeCylinderPressure(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleIndependentBrakeCylinderPressure(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleScriptBrakePressure(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleBrakePipeEfficiency(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleCompressorEfficiency(const TNIPhysicsVehicle* vehicle);
int8_t TNIPhysicsGetVehicleCompressorActive(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleBrakeReleaseSoundFade(const TNIPhysicsVehicle* vehicle);


// Vehicle track state accessors
double TNIPhysicsGetVehicleTrackGradient(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleTrackCurveAngle(const TNIPhysicsVehicle* vehicle);


// Vehicle spec accessors
int8_t TNIPhysicsGetVehicleEngineType(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleMaximumTractiveEffort(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleWheelDiameter(const TNIPhysicsVehicle* vehicle);


// Engine spec accessors
float TNIPhysicsGetVehicleMaxSpeed(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleMaxDCCAcceleration(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleMaxDCCDeceleration(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleMaxVoltage(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetEngineBrakeRatio(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleThrottleAdjustmentRate(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleAxleCount(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleSurfaceArea(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleMovingFrictionCoefficient(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleAirDragCoefficient(const TNIPhysicsVehicle* vehicle);
double TNIPhysicsGetVehicleDrivingWheelWeightRatio(const TNIPhysicsVehicle* vehicle);


// Engine spec flowsize accessors
float TNIPhysicsGetVehicleFlowsize_TrainBrakePipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EpReservoirPipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_No3Pipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_No4Pipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_AuxReservoirVent(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_AuxReservoirToNo3Pipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_AuxReservoirToTrainBrakePipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_AutoBrakeCylinderVent(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_AuxReservoirToAutoBrakeCylinder(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EqualiserMainReservoir(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EqualiserVentHandleOff(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EqualiserVentEmergency(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EqualiserVent(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_No3PipeVent(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_No3PipeMainReservoir(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_Compressor(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_TrainBrakePipeReservoir(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_TrainBrakePipeVent(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_No3PipeToAutoBrakeCylinder(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_EpReservoirPipeToAutoBrakeCylinder(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleFlowsize_MainReservoirEP(const TNIPhysicsVehicle* vehicle);


// Engine spec volume accessors
float TNIPhysicsGetVehicleVolume_Equaliser(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_MainReservoir(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_No3Pipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_TrainBrakePipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_EpReservoirPipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_No4Pipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_AuxReservoir(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_AutoBrakeCylinder(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehicleVolume_IndependentBrakeCylinder(const TNIPhysicsVehicle* vehicle);


// Engine spec pressure accessors
float TNIPhysicsGetVehiclePressure_BrakePipe(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehiclePressure_BrakeInitial(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehiclePressure_BrakeFull(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehiclePressure_IndBrakeFull(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetVehiclePressure_Compressor(const TNIPhysicsVehicle* vehicle);


// Steam engine spec accessors
float TNIPhysicsGetSteamEnginePistonVolumeMin(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEnginePistonVolumeMax(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEnginePistonArea(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEnginePistonLength(const TNIPhysicsVehicle* vehicle);
int TNIPhysicsGetSteamEnginePistonAngularOffsetsCount(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEnginePistonAngularOffset(const TNIPhysicsVehicle* vehicle, int i);

float TNIPhysicsGetSteamEngineTractiveEffortConstant(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineHandBrakeMaxForce(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineTestSpeed(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineTestCutoff(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBurnRate(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBurnRateIdle(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineMinFireTemperature(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineMaxFireTemperature(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFireboxEfficiency(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFireboxHeatingSurfaceArea(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFireboxPlateThickness(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFireboxThermalConductivity(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBlowerEffect(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBlowerMaxFlow(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineStartingCoal(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineIdealCoalMass(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineMaxFireCoalMass(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFuelEnergy(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineFuelSpecificHeatCapacity(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineBoilerVolume(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerVolumeMin(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerVolumeMax(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerFusiblePlugVolume(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSteamChestVolume(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSteamChestMaxFlow(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineValveLapPercent(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineBoilerEfficency(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerEfficencyIdle(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerEfficencyMin(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineBoilerHeatLoss(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineInitialBoilerTemperature(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSuperheatingConstant(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineStartingWater(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineStartingBoilerSteam(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineSafetyValveLowPressure(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSafetyValveLowFlow(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSafetyValveHighPressure(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineSafetyValveHighFlow(const TNIPhysicsVehicle* vehicle);

float TNIPhysicsGetSteamEngineCoalShovelMass(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineWaterInjectorRate(const TNIPhysicsVehicle* vehicle);
float TNIPhysicsGetSteamEngineWaterInjectorRate2(const TNIPhysicsVehicle* vehicle);



#ifdef __cplusplus
}
#endif // __cplusplus


#endif  //  _TNIPhysics_H_
