//=============================================================================
// Name: TNISTLAlloc
// Name: Defines custom allocator for stl classes that make use of the TNI
//       allocation functions. For automatic use declare stl containers using
//       tnistd in place of std, e.g. tnistd::map<int, int>
//=============================================================================
#ifndef _TNISTL_H
#define _TNISTL_H
#pragma once

#define _CRT_SECURE_NO_WARNINGS 1
#define _CRT_SECURE_NO_DEPRECATE 1
#include <vector>
#include <map>
#include <string>
#include <deque>
#include <set>
#include <list>
#include <new>


// Imports from TrainzNativeInterface.h
extern "C" void* TNIAllocMem(size_t size, unsigned int trackingID);
extern "C" void TNIFreeMem(void* memory);



//=============================================================================
// Name: TNISTLAlloc
// Name: Defines custom allocator for stl classes that make use of the TNI
//       allocation functions. For automatic use declare stl containers using
//       tnistd in place of std, e.g. tnistd::map<int, int>
//=============================================================================
template<unsigned int TYPEID, typename T>
struct TNISTLAlloc : public std::allocator<T>
{
  template<typename O>
  struct rebind
  {
    typedef TNISTLAlloc<TYPEID, O> other;
  };

  TNISTLAlloc() { }

  template<typename O>
  TNISTLAlloc(TNISTLAlloc<TYPEID, O>) { }

  // Copy-construct an object at a specified address
	void construct(T* ptr, const T& val)
	{
    // Placement New.
    new(ptr)T(val);
	}

  // Allocate memory for count objects
  T* allocate(std::size_t count)
  {
    return (T *) TNIAllocMem(count * sizeof(T), TYPEID);
  }
  
  #if USE_CPP23
  std::allocation_result<T*> allocate_at_least(size_t count)
  {
    return std::allocation_result<T*>{allocate(count), count};
  }
  #endif // USE_CPP23

  // Deallocate memory at ptr
  void deallocate(T* ptr, std::size_t)
  {
    TNIFreeMem(ptr);
  }
};



//=============================================================================
// Convenience template classes to automatically use TNISTLAlloc
//=============================================================================
namespace tnistd
{
  // jetstd::map
  template<unsigned int TYPEID, class Key, class Val, class Cmp = std::less<Key> >
  class map : public std::map<Key, Val, Cmp, TNISTLAlloc< TYPEID, std::pair<const Key, Val> > >  { };

  // jetstd::multimap
  template<unsigned int TYPEID, class Key, class Val, class Cmp = std::less<Key> >
  class multimap : public std::multimap<Key, Val, Cmp, TNISTLAlloc< TYPEID, std::pair<const Key, Val> > >  { };

  // jetstd::deque
  template<unsigned int TYPEID, class T>
  class deque : public std::deque<T, TNISTLAlloc< TYPEID, T > > { };

  // jetstd::set
  template<unsigned int TYPEID, class T, class Cmp = std::less<T> >
  class set : public std::set<T, Cmp, TNISTLAlloc< TYPEID, T > > { };

  // jetstd::vector
  template<unsigned int TYPEID, class T>
  class vector : public std::vector<T, TNISTLAlloc< TYPEID, T > > { };

  // jetstd::list
  template<unsigned int TYPEID, class T>
  class list : public std::list<T, TNISTLAlloc< TYPEID, T > > { };
}


#endif // _TNISTL_H

