//=============================================================================
// File: TNIStream.h
// Desc: Declares a data stream class for use in TNI libraries.
//=============================================================================
#ifndef _TNISTREAM_H_
#define _TNISTREAM_H_
#pragma once

#include "trainznativeinterface.h"


//=============================================================================
// COMPILER OPTIONS
//
// Despite being written in C++, this interface is considered portable and 
// third-party developers are able to access the trainznativeinterface.dll 
// from any native language which is capable of building a DLL by building an
// alternative header in that language.
// 
// Exceptions must not be thrown in DLL code.
//=============================================================================
#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

  


//=============================================================================
// Name: TNIStreamInitData
// Desc: An initialisation structure used when creating a custom TNIStream. 
//       This structure is not a TNIObject and may be safely constructed by the
//       third-party developer.
//=============================================================================
class TNIStreamInitData
{
protected:

  uint32_t      m_structVersion;            // The struct format version, set during construction.


public:
  
  // The StreamShutdownCall function pointer type.
  typedef void (*StreamShutdownCall)(void* streamCookie);

  // The StreamWriteCall function pointer type.
  typedef bool (*StreamWriteCall)(void* streamCookie, const void* buffer, size_t bufferLength);
  
  // The StreamReadCall function pointer type.
  typedef bool (*StreamReadCall)(void* streamCookie, void* o_buffer, size_t bufferLength);
  
  // The StreamResizeCall function pointer type.
  typedef bool (*StreamResizeCall)(void* streamCookie, size_t newLength);
  
  // The StreamSeekCall function pointer type.
  typedef bool (*StreamSeekCall)(void* streamCookie, int seekDirection, int seekOffset);
  
  // The StreamTellCall function pointer type.
  typedef size_t (*StreamTellCall)(void* streamCookie);
  
  // The StreamGetSizeCall function pointer type.
  typedef size_t (*StreamGetSizeCall)(void* streamCookie);
  
  // The StreamStoreObjectCall function pointer type.
  typedef int32_t (*StreamStoreObjectCall)(void* streamCookie, const TNIObject* object);
  
  // The StreamRetrieveObjectCall function pointer type.
  typedef const TNIObject* (*StreamRetrieveObjectCall)(void* streamCookie, int32_t object);


  void*                     m_cookie;                       // User storage. The value specified here is passed to the function calls.
  StreamShutdownCall        m_streamShutdownCall;           // A pointer to the function which is called during shutdown of this stream.
  StreamWriteCall           m_streamWriteCall;              // A pointer to the function which services Write calls for this stream.
  StreamReadCall            m_streamReadCall;               // A pointer to the function which services Read calls for this stream.
  StreamResizeCall          m_streamResizeCall;             // A pointer to the function which services Resize calls for this stream.
  StreamSeekCall            m_streamSeekCall;               // A pointer to the function which services Seek calls for this stream.
  StreamTellCall            m_streamTellCall;               // A pointer to the function which services Tell calls for this stream.
  StreamGetSizeCall         m_streamGetSizeCall;            // A pointer to the function which services GetSize calls for this stream.
  StreamStoreObjectCall     m_streamStoreObjectCall;        // A pointer to the function which services Store Object calls for this stream.
  StreamRetrieveObjectCall  m_streamRetrieveObjectCall;     // A pointer to the function which services Retrieve Object calls for this stream.


  // The default constructor; this will initialise all fields appropriately, such that
  // the developer need only specify fields which they wish to override.
  inline TNIStreamInitData(void)
    : m_structVersion(0)
    , m_cookie(NULL)
    , m_streamShutdownCall(NULL)
    , m_streamWriteCall(NULL)
    , m_streamReadCall(NULL)
    , m_streamResizeCall(NULL)
    , m_streamSeekCall(NULL)
    , m_streamTellCall(NULL)
    , m_streamGetSizeCall(NULL)
    , m_streamStoreObjectCall(NULL)
    , m_streamRetrieveObjectCall(NULL)
  {
  }

};



//=============================================================================
// Name: TNIStream
// Desc: A simple data stream. Can be dynamically resizeable or not.
//=============================================================================
class TNIStream : public TNIObject
{
protected:
};


#if !TRAINZ_NATIVE_INTERFACE_CPP

//=============================================================================
// Name: TNIAllocStream
// Desc: Allocates a TNIStream object.
// Parm: bufferLen - The initial buffer length.
// Parm: autoResizeAmount - The amount to increase the buffer length by when it
//       is full. This is done when a write operation would otherwise fail
//       because the buffer is full. A value of zero indiciates that the buffer
//       will not automatically resize.
// Retn: The newly created TNIStream object.
//=============================================================================
TNIStream* TNIAllocStream(uint32_t bufferLen, uint32_t autoResizeAmount = 0);


//=============================================================================
// Name: TNIAllocCustomStream
// Desc: Allocates a TNIStream object given an initialiser. This form is used
//       to wrap external stream functionality into the TNIStream interface.
// Parm: streamInitData - The initialiser which describes the internal
//       implementation of the constructed TNIStream object.
// Retn: The newly created TNIStream object.
//=============================================================================
TNIStream* TNIAllocCustomStream(const TNIStreamInitData& streamInitData);


//=============================================================================
// Name: TNICloseCustomStream
// Desc: Closes a TNIStream object allocated by TNIAllocCustomStream. This will
//       ensure the StreamShutdownCall is correctly called without the need to
//       release every reference to the stream.
// Parm: stream - The stream to close/shutdown.
//=============================================================================
void TNICloseCustomStream(TNIStream* stream);



//=============================================================================
//=============================================================================
bool TNISerialiseToStream(TNIContext* context, const TNIObject* srcObject, TNIStream* dstStream, bool (*helperCallback)(TNIContext* context, const TNIObject* object, TNIStream* dstStream) = NULL);


//=============================================================================
//=============================================================================
TNIObject* TNISerialiseFromStream(TNIContext* context, TNIStream* srcStream, TNIObject* (*helperCallback)(TNIContext* context, int32_t objectClass, TNIStream* srcStream) = NULL);


//=============================================================================
// Name: TNIStreamResize
// Desc: Resizes the buffer inside a TNIStream object
// Parm: stream - the TNIStream to perform the operation on.
// Parm: newLen - The new buffer length
//=============================================================================
void TNIStreamResize(TNIStream* stream, uint32_t newLen);


//=============================================================================
// Name: TNIStreamSeek/TNIStreamSeekFromEnd
// Desc: Seek operations move the internal buffer position to that indiciated.
//       A seek will fail if the position is out of the buffer range, seek
//       operations will not cause the buffer to resize.
// Parm: stream - the TNIStream to perform the operation on.
// Parm: pos - The new buffer pos.
// Retn: bool - true if the buffer position was altered as requested.
//=============================================================================
bool TNIStreamSeek(TNIStream* stream, uint32_t pos);
bool TNIStreamSeekFromEnd(TNIStream* stream, uint32_t pos);


//=============================================================================
// Name: TNIStreamTell
// Desc: Returns the current cursor position of the specified stream, as 
//       suitable for passing to TNIStreamSeek(). A stream begins at position
//       zero (0). 
// Parm: stream - the TNIStream to perform the operation on.
// Retn: size_t - The current cursor position, in bytes.
//=============================================================================
size_t TNIStreamTell(TNIStream* stream);


//=============================================================================
// Name: TNIStreamGetSize
// Desc: Returns the current size of the specified stream.
// Parm: stream - the TNIStream to perform the operation on.
// Retn: size_t - The current size, in bytes.
//=============================================================================
size_t TNIStreamGetSize(TNIStream* stream);



//=============================================================================
// Name: TNIStreamCopy
// Desc: Copy the specified number of bytes from the source stream to the
//       destination stream.
// Parm: dstStream - The stream to which the data is written.
// Parm: srcStream - The stream from which the data is read.
// Parm: copyLength - The number of bytes to copy.
// Retn: bool - True if the copy was successful, or false if the copy failed
//       for any reason (including EOF on read.)
//=============================================================================
bool TNIStreamCopy(TNIStream* dstStream, TNIStream* srcStream, size_t copyLength);



//=============================================================================
// Name: TNIStreamWrite****/TNIStreamWrite*****Array
// Desc: Write operation for various data types. The operations will write data
//       to the internal buffer. If there is insufficient space the buffer will
//       either be resized (if allowed) or the operation will fail.
// Parm: stream - The TNIStream to perform the operation on.
// Parm: value - The data to copy to the stream.
// Parm: count - For array writes only, indicates the length of the array.
// Retn: bool - true if the operation was successful and the data was written.
//=============================================================================

bool TNIStreamWriteFloat(TNIStream* stream, float value);
bool TNIStreamWriteFloatArray(TNIStream* stream, const float value[], int count);

bool TNIStreamWriteFloat64(TNIStream* stream, double value);
bool TNIStreamWriteFloat64Array(TNIStream* stream, const double value[], int count);

bool TNIStreamWriteInt64(TNIStream* stream, int64_t value);
bool TNIStreamWriteInt64Array(TNIStream* stream, const int64_t value[], int count);

bool TNIStreamWriteInt32(TNIStream* stream, int32_t value);
bool TNIStreamWriteInt32Array(TNIStream* stream, const int32_t value[], int count);

bool TNIStreamWriteInt16(TNIStream* stream, int16_t value);
bool TNIStreamWriteInt16Array(TNIStream* stream, const int16_t value[], int count);

bool TNIStreamWriteInt8(TNIStream* stream, int8_t value);
bool TNIStreamWriteInt8Array(TNIStream* stream, const int8_t value[], int count);

bool TNIStreamWriteString(TNIStream* stream, const TNIString* string);

bool TNIStreamWriteAssetID(TNIStream* stream, const TNIAssetID* assetId);

int32_t TNIStreamStoreObjectReference(TNIStream* stream, const TNIObject* object);
bool TNIStreamWriteTNIObjectReference(TNIStream* stream, const TNIObject* object);



//=============================================================================
// Name: TNIStreamRead****/TNIStreamRead*****Array
// Desc: Read operation for various data types. The operations will read data
//       from the internal buffer. If the read operation would push the buffer
//       position past the end of the buffer the operation will fail and no
//       read will take place.
// Parm: stream - The TNIStream to perform the operation on.
// Parm: failureSignal - Single reads only, value to return if the op fails.
// Parm: value - Array reads only, The array to copy the stream data to.
// Parm: count - Array reads only, indicates the length of the array.
// Retn: bool - Array reads only, true if the operation was successful.
//=============================================================================


float TNIStreamReadFloat(TNIStream* stream, float failureSignal = -1.0);
bool TNIStreamReadFloatArray(TNIStream* stream, float value[], int count);

double TNIStreamReadFloat64(TNIStream* stream, double failureSignal = -1.f);
bool TNIStreamReadFloat64Array(TNIStream* stream, double value[], int count);

int64_t TNIStreamReadInt64(TNIStream* stream, int64_t failureSignal = -1);
bool TNIStreamReadInt64Array(TNIStream* stream, int64_t value[], int count);

int32_t TNIStreamReadInt32(TNIStream* stream, int32_t failureSignal = -1);
bool TNIStreamReadInt32Array(TNIStream* stream, int32_t value[], int count);

int16_t TNIStreamReadInt16(TNIStream* stream, int16_t failureSignal = -1);
bool TNIStreamReadInt16Array(TNIStream* stream, int16_t value[], int count);

int8_t TNIStreamReadInt8(TNIStream* stream, int8_t failureSignal = -1);
bool TNIStreamReadInt8Array(TNIStream* stream, int8_t value[], int count);

TNIString* TNIStreamReadString(TNIStream* stream);

TNIAssetID* TNIStreamReadAssetID(TNIStream* stream);

const TNIObject* TNIStreamRetrieveObjectReference(TNIStream* stream, int32_t object);

const TNIObject* TNIStreamReadTNIObjectReference(TNIStream* stream);


// ============================================================================
// HELPER FUNCTIONS
// ============================================================================
// The "cast" functions return the passed-in object cast to the appropriate
// type, or nullptr if the passed-in object is not of the appropriate type.
//
// In the "release" forms, the parameter object is effectively released.

const TNIStream* TNICastStream(const TNIObject* object);

const TNIStream* TNICastStreamRelease(const TNIObject* object);


#endif // !TRAINZ_NATIVE_INTERFACE_CPP


//=============================================================================
// END COMPILER OPTIONS
//=============================================================================
#ifdef __cplusplus
}
#endif // __cplusplus

#endif  //  _TNISTREAM_H_
