//=============================================================================
// File: trainznativeinterface.h
// Desc: This C++ header provides an interface to the trainznativeinterface.dll
//       which is included in the Trainz /bin folder. This is a public 
//       interface for third-party developers to write native extensions to
//       Trainz.
//       Functions and classes in this file provide interoperability primitives
//       which allow data to be passed between Trainz and third-party DLLs.
//=============================================================================
#ifndef _TRAINZNATIVEINTERFACE_H
#define _TRAINZNATIVEINTERFACE_H
#pragma once


//=============================================================================
// Don't use Windows STL debugging.
#define _SECURE_SCL 0
#define _ITERATOR_DEBUG_LEVEL 0


#include <cstddef>
#include <atomic>
#include <cstdint>


//=============================================================================
// Compiler error for build configurations which do not support TNI.
#if TRAINZ_DONT_INCLUDE_TRAINZNATIVEINTERFACE
  #error Somebody included this header at an inappropriate moment.
#endif // TRAINZ_DONT_INCLUDE_TRAINZNATIVEINTERFACE


//=============================================================================
// STANDARD TYPES
//=============================================================================
#ifdef _MSC_VER
  typedef signed __int64 int64_t;
  typedef signed __int32 int32_t;
  typedef signed __int16 int16_t;
  typedef signed __int8  int8_t;
  typedef unsigned __int64 uint64_t;
  typedef unsigned __int32 uint32_t;
  typedef unsigned __int16 uint16_t;
  typedef unsigned __int8  uint8_t;
  #define TNI_ENTRYPOINT_DECL __declspec(dllexport)
#else // !_MSC_VER
  #include <stdint.h>
  #define TNI_ENTRYPOINT_DECL 
#endif // !_MSC_VER



//=============================================================================
// COMPILER OPTIONS
//
// Despite being written in C++, this interface is considered portable and 
// third-party developers are able to access the trainznativeinterface.dll 
// from any native language which is capable of building a DLL by building an
// alternative header in that language.
// 
// Exceptions must not be thrown in DLL code.
//=============================================================================
#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

#ifdef _MSC_VER
#pragma warning(disable:4510)
#pragma warning(disable:4610)
#pragma warning(disable:4068) // warning C4068 : unknown pragma 'mark'
#endif // _MSC_VER



//=============================================================================
// TNI OBJECT TYPES
//=============================================================================
enum
{
  TNI_CLASS_NONE = 0,
  TNI_CLASS_OBJECT = 1,
  TNI_CLASS_STRING = 2,
  TNI_CLASS_ARRAY = 3,
  TNI_CLASS_CONTEXT = 4,
  TNI_CLASS_LIBRARY = 5,
  TNI_CLASS_STREAM = 6,
  TNI_CLASS_GRAPHICS = 7,
  TNI_CLASS_VECTOR = 8,
  TNI_CLASS_ASSETID = 9,
  TNI_CLASS_LABEL = 10,
  TNI_CLASS_BUFFER = 11,
  TNI_CLASS_SOUP = 12,
  TNI_CLASS_INTEGER = 13,
  TNI_CLASS_FLOAT = 14,
  TNI_CLASS_COORDINATES = 15,
  TNI_CLASS_HANDLE = 16,
  TNI_CLASS_PHYSICS_VEHICLE = 17,
  TNI_CLASS_PHYSICS_VEHICLE_FRAME_SNAPSHOT = 18,
  TNI_CLASS_PHYSICS_ENGINE_SPEC = 19,
  TNI_CLASS_PHYSICS_STEAMENGINE_SPEC = 20,
  TNI_CLASS_GAME_OBJECT_ID = 21,
};



//=============================================================================
// TNI OBJECT FORWARD DECLARATIONS
//=============================================================================
class TNIObject;
class TNIString;
class TNIInteger;
class TNIFloat;
class TNIAssetID;
class TNIArray;
class TNIContext;
class TNILibrary;
class TNIStream;
class TNIGraphics;
class TNIBuffer;
class TNILabel;
class TNICoordinates;
class TNIGameObjectID;



//=============================================================================
// TNI IMPLEMENTATION FORWARD DECLARATIONS
//=============================================================================
class TNIHandleTarget;



//=============================================================================
// TNI OBJECT SCOPE MANAGEMENT
//
// TNIObject-derived classes use reference counting. Each object instance is
// allocated with a reference count of one. Functions which take an object as
// a parameter may use that object within their scope, however if they wish to
// keep a reference beyond the lifetime of the function, they must use
// TNIReference() to increment the reference count of the object, and call
// TNIRelease() to decrement the reference count once the object is no longer
// required.
// 
// As a result of the above, the caller of a function need not take an
// additional reference on any objects which are passed as parameters to the
// function, and the reference count is considered to be unchanged across the
// function call - it is still the callers responsibility to release any
// existing references.
// 
// Any function which returns a reference-counted object must not release it
// prior to the return. It is the responsibility of the caller to release the
// final reference count of the return value.
//
// Any parameter or return value declared as a TNIObject* must also safely
// handle a NULL value. The NULL value may be considered a parameter error,
// however it must be handled appropriately, preventing any exception or
// data corruption.
//
// Some specialised functions may release a passed-in TNIObject* parameter, in
// opposition to the standard behaviour defined above. These functions will
// include the term "Release" in their function name. Examples of this usage
// include TNIRelease() and TNICastStringRelease().
//=============================================================================


//=============================================================================
// Name: TNIReference
// Desc: Add a reference to the specified object. The object will be released
//       when the reference count drops to zero.
// Parm: object - The object to which we will add a reference. It is safe to
//       pass a NULL value here.
// Retn: const TNIObject* - The same object passed in, for convenience.
//=============================================================================
const TNIObject* TNIReference(const TNIObject* object);



//=============================================================================
// Name: TNIRelease
// Desc: Remove a reference to the specified object. The object will be
//       released when the reference count drops to zero. TNIRelease() should
//       be called exactly once for each call to TNIReference() and once for
//       any function return value (for example, from TNIAllocString().)
// Parm: object - The object from which we will remove a reference. It is safe
//       to pass a NULL value here.
//=============================================================================
void TNIRelease(const TNIObject* object);



//=============================================================================
// Name: TNIObject
// Desc: The base object from which all trainznativeinterface objects derive. 
//       All instances of TNIObject-derived classes are created through the 
//       functions present in this file. Never attempt to construct one of
//       these objects yourself.
//       All object data is private. Use the accessor functions (below) to
//       manipulate the objects.
//=============================================================================
class TNIObject
{
protected:

  // Prevent accidental construction.
  TNIObject(uint32_t objectType);

  
protected:
  
  friend void TNIRelease(const TNIObject* object);
  ~TNIObject(void);
  void operator=(const TNIObject& ) {}
};



//=============================================================================
// Name: TNIString
// Desc: A UTF-8 encoded text string.
//=============================================================================
class TNIString : public TNIObject
{
};



//=============================================================================
// Name: TNIInteger
// Desc: A single integer value.
//=============================================================================
class TNIInteger : public TNIObject
{
};



//=============================================================================
// Name: TNIFloat
// Desc: A single floating-point value.
//=============================================================================
class TNIFloat : public TNIObject
{
};



//=============================================================================
// Name: TNIAssetID
// Desc: An Asset ID (typically, but not necessarily, a KUID).
//=============================================================================
class TNIAssetID : public TNIObject
{
};



//=============================================================================
// Name: TNIArray
// Desc: A vector of objects.
//=============================================================================
class TNIArray : public TNIObject
{
};



//=============================================================================
// Name: TNISoup
// Desc: A key-value database.
//=============================================================================
class TNISoup : public TNIObject
{
};



//=============================================================================
// Name: TNIVector
// Desc: A vector of floating-point values.
//=============================================================================
class TNIVector : public TNIObject
{
};



//=============================================================================
// Name: TNICoordinates
// Desc: A single position within a Trainz route.
//=============================================================================
class TNICoordinates : public TNIObject
{
};



//=============================================================================
// Name: TNIHandle
// Desc: A reference to a native object of some form that TNI does not support
//       explicitly. 
//=============================================================================
class TNIHandle : public TNIObject
{
};



//=============================================================================
// MEMORY ALLOCATION
//=============================================================================



//=============================================================================
// Name: TNIAllocMem
// Desc: Allocate memory from the shared memory pool. This allocation may be
//       safely accessed across any DLL. TNIFreeMem() must be called exactly
//       once to release the allocated memory.
// Parm: size - The required size, in bytes.
// Parm: trackingID - A four-char-code identity for the allocation's purpose.
//       Used for debugging leaks.
// Retn: void* - An allocation of at least 'size' contiguous bytes of memory.
//       The returned memory space is zero-initialised. Access beyond the
//       requested bounds is illegal. On allocation failure, or if the input
//       size is zero, the function returns NULL.
//=============================================================================
void* TNIAllocMem(size_t size, unsigned int trackingID = '\?\?\?\?');


//=============================================================================
// Name: TNIFreeMem
// Desc: Free memory that was allocated from the shared memory pool. This must
//       be called exactly once for each usage of TNIAllocMem(). After this 
//       call, the memory is considered deallocated and the same pointer may be
//       returned from a future call to TNIAllocMem(). Access to the allocation
//       after calling TNIFreeMem() is illegal.
// Parm: memory - The memory pointer as returned from a call to TNIAllocMem.
//       It is also safe to pass a NULL value here.
//=============================================================================
void TNIFreeMem(void* memory);


//=============================================================================
// Name: TNIThreadCreate
// Desc: Creates a new thread of execution, calling the specified entrypoint
//       function. The thread terminates when the entrypoint function returns.
// Note: Unnecessary thread creation should be avoided for performance reasons.
// Note: It is not possible for Trainz to forcibly terminate a thread. It is
//       the caller's responsibility to communicate with the thread and ensure
//       that it shuts itself down at the appropriate time.
// Parm: threadDebugLabel - A label used to identify this thread for debugging
//       purposes. Should be meaningful but is not required to be truly unique.
// Parm: cookie - A cookie which is passed to the entrypoint function. Not
//       parsed by the TNI API.
// Parm: entrypoint - The entrypoint function.
//=============================================================================
void TNIThreadCreate(const TNILabel* threadDebugLabel, void* cookie, void (*entrypoint)(void*));



//=============================================================================
// UTILITY FUNCTIONS
//=============================================================================



//=============================================================================
// Name: TNIThreadSleep
// Desc: Called on a thread created via TNIThreadCreate, this function
//       efficiently blocks execution for approximately the specified period.
// Parm: sleepDurationMs - The number of milliseconds for which this function
//       should block execution. This is advisory only; the actual sleep may
//       run for longer or shorter than the requested interval.
// Note: This function should not be called on Trainz' own threads, only on
//       threads created TNIThreadCreate().
//=============================================================================
void TNIThreadSleep(unsigned int sleepDurationMs);



//=============================================================================
// END COMPILER OPTIONS
//=============================================================================
#ifdef __cplusplus
}
#endif // __cplusplus

#endif // _TRAINZNATIVEINTERFACE_H
