// ============================================================================
// File: trainznativeinterface.hpp
// Desc: C++11 specific helper wrappers around the (language-agnostic) Trainz
//       Native Interface API. This functionality is provided to simplify 
//       programming TNI DLLs in the C++11 language.
// ============================================================================
#ifndef _TRAINZNATIVEINTERFACE_HPP
#define _TRAINZNATIVEINTERFACE_HPP
#pragma once


#ifndef TRAINZ_NATIVE_INTERFACE_CPP
  #define TRAINZ_NATIVE_INTERFACE_CPP 1
#endif // !defined(TRAINZ_NATIVE_INTERFACE_CPP)
#include "trainznativeinterface.h"
#include <utility>


// ============================================================================
// Static-Assert Helper.
// ============================================================================
template<class> class always_false : public std::false_type {};


// ============================================================================
// ============================================================================
template <class T> class TNIRefBase;
template <class T> class TNIRef;


// ============================================================================
// internal use only.
// ============================================================================
class TNIRefTypelessBase
{
protected:
  
  void LockMutex(void);
  void UnlockMutex(void);
};


// ============================================================================
// internal use only.
// ============================================================================
template <class T> class TNIRefBase : public TNIRefTypelessBase
{
public:

  explicit TNIRefBase(T* obj) : m_object(obj) {}

  bool operator==(const TNIRefBase<typename std::remove_const<T>::type>& obj) const;
  bool operator==(const TNIRefBase<const T>& obj) const;
  bool operator==(const T* obj) const;
  bool operator!=(const TNIRefBase<typename std::remove_const<T>::type>& obj) const;
  bool operator!=(const TNIRefBase<const T>& obj) const;
  bool operator!=(const T* obj) const;

  bool operator<(const TNIRefBase<T>& obj) const { return m_object < obj.m_object; }
  
  operator bool(void) const;
  bool operator !(void) const;
  
  // operator T* (void) const; -- this is a nice operator to have around, but it also substantially increases the risk of messing up if you mix C and C++ coding styles.
  T* c_obj(void) const;
  T* c_obj_inc(void) const;
  T* operator ->(void) const;
  
  
protected:

  friend class TNIRef<T>;
  
  T* m_object;
};


// ============================================================================
// ============================================================================
template <class T> bool operator==(const T* lhs, const TNIRefBase<T>& rhs)
{
  return (lhs == rhs.c_obj());
}


// ============================================================================
// ============================================================================
template <class T> bool operator!=(const T* lhs, const TNIRefBase<T>& rhs)
{
  return (lhs != rhs.c_obj());
}


// ============================================================================
// Name: TNIRef
// Desc: A templated class which wraps any pointer to a TNIObject. Constructing
//       from an TNIObject* will take ownership of the object's reference. The 
//       reference is automatically released when the TNIRef object is 
//       overwritten or destroyed.
// ============================================================================
template <class T> class TNIRef : public TNIRefBase<T>
{
public:
  
  TNIRef(void);
  TNIRef(std::nullptr_t obj);
  //TNIRef(T* obj);
  TNIRef(const TNIRef<T>& obj);
  TNIRef(TNIRef<T>&& obj);
  TNIRef(const TNIRefBase<T>& obj);
  template <class X> TNIRef(const TNIRefBase<X>& obj);
  //TNIRef(TNIRefBase<T>&& obj);
  ~TNIRef(void);
  
  const TNIRef<T>& operator=(const TNIRef<T>& obj);
  const TNIRef<T>& operator=(TNIRef<T>&& obj);
  const TNIRef<T>& operator=(const TNIRefBase<T>& obj);
  //const TNIRef<T>& operator=(TNIRefBase<T>&& obj);
  const TNIRef<T>& operator=(const std::nullptr_t& obj);
  
  
  static TNIRef<T> Swallow(T* obj);
  static TNIRef<T> Swallow(const TNIRefBase<T>& obj);
  static TNIRef<T> Reference(T* obj);
  static TNIRef<T> Reference(const TNIRefBase<T>& obj);
};


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(void)
  : TNIRefBase<T>(nullptr)
{
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(std::nullptr_t obj)
  : TNIRefBase<T>(nullptr)
{
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T> TNIRef<T>::Swallow(T* obj)
{
  TNIRef<T> ret;
  ret.m_object = obj;
  return ret;
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T> TNIRef<T>::Swallow(const TNIRefBase<T>& obj)
{
  static_assert(always_false<T>::value, "TNIRef<T>::Swallow> don't call this with a TNIRef");
  return obj;
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T> TNIRef<T>::Reference(T* obj)
{
  TNIRef<T> ret;
  ::TNIReference(obj);
  ret.m_object = obj;
  return ret;
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T> TNIRef<T>::Reference(const TNIRefBase<T>& obj)
{
  return obj;
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(const TNIRef<T>& obj)
  : TNIRefBase<T>(nullptr)
{
  this->LockMutex();
  
  ::TNIReference(obj.c_obj());
  this->m_object = obj.c_obj();
  
  this->UnlockMutex();
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(TNIRef<T>&& obj)
  : TNIRefBase<T>(nullptr)
{
  this->LockMutex();
  
  this->m_object = obj.c_obj();
  obj.m_object = nullptr;
  
  this->UnlockMutex();
}


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(const TNIRefBase<T>& obj)
  : TNIRefBase<T>(nullptr)
{
  this->LockMutex();
  
  ::TNIReference(obj.c_obj());
  this->m_object = obj.c_obj();
  
  this->UnlockMutex();
}


// ============================================================================
// ============================================================================
template <class T> template <class X> TNIRef<T>::TNIRef(const TNIRefBase<X>& obj)
  : TNIRefBase<T>(nullptr)
{
  this->LockMutex();
  
  ::TNIReference(obj.c_obj());
  this->m_object = obj.c_obj();
  
  this->UnlockMutex();
}

/* not valid, since TNIRefBase<> might be a TNIRefParam<>
// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::TNIRef(TNIRefBase<T>&& obj)
  : TNIRefBase<T>(obj.m_object)
{
  obj.m_object = nullptr;
}
*/


// ============================================================================
// ============================================================================
template <class T> TNIRef<T>::~TNIRef(void)
{
  ::TNIRelease(this->m_object);
}


// ============================================================================
// ============================================================================
template <class T> const TNIRef<T>& TNIRef<T>::operator=(const TNIRef<T>& obj)
{
  this->LockMutex();
  
  auto released = this->m_object;
  TNIReference(obj.c_obj());
  this->m_object = obj.c_obj();
  
  this->UnlockMutex();
  
  TNIRelease(released);
  
  return *this;
}


// ============================================================================
// ============================================================================
template <class T> const TNIRef<T>& TNIRef<T>::operator=(TNIRef<T>&& obj)
{
  this->LockMutex();
  
  if (obj.m_object == this->m_object)
  {
    this->UnlockMutex();
    return *this;
  }
  
  auto released = this->m_object;
  this->m_object = obj.m_object;
  obj.m_object = nullptr;
  
  this->UnlockMutex();
  
  TNIRelease(released);
  
  return *this;
}


// ============================================================================
// ============================================================================
template <class T> const TNIRef<T>& TNIRef<T>::operator=(const TNIRefBase<T>& obj)
{
  this->LockMutex();
  
  auto released = this->m_object;
  TNIReference(obj.c_obj());
  this->m_object = obj.c_obj();
  
  this->UnlockMutex();
  
  TNIRelease(released);
  
  return *this;
}


/* Not valid, since TNIRefBase<> might be TNIRefParam<>
// ============================================================================
// ============================================================================
template <class T> const TNIRef<T>& TNIRef<T>::operator=(TNIRefBase<T>&& obj)
{
  if (this == &obj)
    return *this;
  
  TNIRelease(this->m_object);
  this->m_object = obj.m_object;
  obj.m_object = nullptr;
  
  return *this;
}
*/


// ============================================================================
// ============================================================================
template <class T> const TNIRef<T>& TNIRef<T>::operator=(const std::nullptr_t& obj)
{
  this->LockMutex();
  auto released = this->m_object;
  this->m_object = nullptr;
  this->UnlockMutex();
  
  TNIRelease(released);
  
  return *this;
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator==(const TNIRefBase<typename std::remove_const<T>::type>& obj) const
{
  return (this->m_object == obj.c_obj());
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator==(const TNIRefBase<const T>& obj) const
{
  return (this->m_object == obj.c_obj());
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator==(const T* obj) const
{
  return (this->m_object == obj);
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator!=(const TNIRefBase<typename std::remove_const<T>::type>& obj) const
{
  return (this->m_object != obj.c_obj());
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator!=(const TNIRefBase<const T>& obj) const
{
  return (this->m_object != obj.c_obj());
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator!=(const T* obj) const
{
  return (this->m_object != obj);
}


// ============================================================================
// ============================================================================
template <class T> TNIRefBase<T>::operator bool(void) const
{
  return (this->m_object != nullptr);
}


// ============================================================================
// ============================================================================
template <class T> bool TNIRefBase<T>::operator !(void) const
{
  return (this->m_object == nullptr);
}


// ============================================================================
// ============================================================================
template <class T> T* TNIRefBase<T>::c_obj(void) const
{
  return this->m_object;
}


// ============================================================================
// ============================================================================
template <class T> T* TNIRefBase<T>::c_obj_inc(void) const
{
  ::TNIReference(this->m_object);
  return this->m_object;
}


// ============================================================================
// ============================================================================
template <class T> T* TNIRefBase<T>::operator-> (void) const
{
  return this->m_object;
}


// ============================================================================
// ============================================================================
template <class T> class TNIRefParam : public TNIRefBase<T>
{
public:

  TNIRefParam(void);
  TNIRefParam(std::nullptr_t obj);
  TNIRefParam(T* obj);
  TNIRefParam(const TNIRefBase<T>& obj);
  template <class X> TNIRefParam(const TNIRefBase<X>& obj);
  ~TNIRefParam(void);

  operator T* (void) const;
};


// ============================================================================
// ============================================================================
template <class T> TNIRefParam<T>::TNIRefParam(void)
  : TNIRefBase<T>(nullptr)
{
}


// ============================================================================
// ============================================================================
template <class T> TNIRefParam<T>::TNIRefParam(std::nullptr_t obj)
  : TNIRefBase<T>(nullptr)
{
}


// ============================================================================
// ============================================================================
template <class T> TNIRefParam<T>::TNIRefParam(T* obj)
  : TNIRefBase<T>(obj)
{
  // We do not own a reference to the object.
}


// ============================================================================
// ============================================================================
template <class T> TNIRefParam<T>::TNIRefParam(const TNIRefBase<T>& obj)
  : TNIRefBase<T>(obj.c_obj())
{
  // We do not own a reference to the object.
}


// ============================================================================
// ============================================================================
template <class T> template <class X> TNIRefParam<T>::TNIRefParam(const TNIRefBase<X>& obj)
  : TNIRefBase<T>(obj.c_obj())
{
  // We do not own a reference to the object.
}


// ============================================================================
// ============================================================================
template <class T> TNIRefParam<T>::~TNIRefParam(void)
{
  // We do not release our reference, because we do not own it.
}


// ============================================================================
// A lot safer than with TNIRef<>, so the benefits outweigh the risks.
// ============================================================================
template <class T> TNIRefParam<T>::operator T* (void) const
{
  return this->m_object;
}



// ============================================================================
// Name: TNIHandleTarget
// Desc: The basetype for any object which can be the target of a TNIHandle.
// ============================================================================
class TNIHandleTarget
{
public:
  
  virtual void TNIAddReference(void) const = 0;
  virtual void TNIRemoveReference(void) const = 0;
};


// ============================================================================
// Name: TNIReference
// Desc: A templated override which allows the function to return the same
//       object type that is passed in.
// ============================================================================
template <class T> T* TNIReference(T* object)
{
  (void)TNIReference((const TNIObject*)object);
  return object;
}


// ============================================================================
// Name: TNIReference
// Desc: A templated override which allows the function to return the same
//       object type that is passed in.
// ============================================================================
template <class T> T* TNIReference(const TNIRef<T>& object)
{
  T* obj = (T*)object;
  (void)TNIReference(obj);
  return obj;
}


// ============================================================================
// Name: TNIRelease
// Desc: This allows a TNIRef<> to be released at a specified moment.
// ============================================================================
template <class T> inline void TNIRelease(TNIRef<T>& io_released)
{
  io_released = nullptr;
  //static_assert(always_false<T>::value, "TNIRelease> don't call this with a TNIRef - they are self-releasing");
}


// ============================================================================
// Name: TNIRelease
// Desc: This allows a TNIRef<> to be released at a specified moment.
// ============================================================================
template <class T> inline void TNIRelease(const TNIRefParam<T>& io_released)
{
  static_assert(always_false<T>::value, "TNIRelease> don't call this with a TNIRefParam");
}


// ============================================================================
// Name: TNISwallowReference
// Desc: Creates a TNIRef<> with the appropriate type to contain 'object', then
//       consumes the object reference.
// Parm: object - The object whose reference will be consumed.
// Retn: TNIRef<T> - A new TNIRef<> to which the object reference is
//       transferred.
// ============================================================================
template <class T> inline TNIRef<T> TNISwallowReference(T* object)
{
  return TNIRef<T>::Swallow(object);
}


// ============================================================================
// Name: TNISwallowReference
// Desc: Blocks illegal behaviour.
// ============================================================================
template <class T> inline TNIRef<T> TNISwallowReference(const TNIRefBase<T>& object)
{
  static_assert(always_false<T>::value, "TNIRef<T>::Swallow> don't call this with a TNIRef");
  return nullptr;
}


// ============================================================================
// Name: TNIAddReference
// Desc: Creates a TNIRef<> with the appropriate type to contain 'object', then
//       copies the object reference.
// Parm: object - The object whose reference will be copied.
// Retn: TNIRef<T> - A new TNIRef<> to which the object reference is
//       copied.
// ============================================================================
template <class T> inline TNIRef<T> TNIAddReference(T* object)
{
  return TNIRef<T>::Reference(object);
}

// ============================================================================
// ============================================================================
template <class T> inline TNIRef<T> TNIAddReference(const TNIRefBase<T>& object)
{
  return TNIRef<T>::Reference(object);
}



#endif // _TRAINZNATIVEINTERFACE_HPP
